import asyncio
import os
import json
import shutil
import re

from datetime import datetime
from serial import Serial

import numpy as np
import matplotlib.pyplot as plt

from py_pli.pylib import VUnits
from py_pli.pylib import GlobalVar
from py_pli.pylib import send_gc_event

from predefined_tasks.common.helper import send_to_gc
from predefined_tasks.common.node_io import FMBAnalogOutput
from predefined_tasks.common.node_io import FMBDigitalOutput
from predefined_tasks.common.node_io import EEFAnalogInput
from predefined_tasks.common.node_io import EEFAnalogOutput
from predefined_tasks.common.node_io import EEFDigitalOutput

from config_enum import hal_enum as hal_config
from config_enum import focus_mover_enum as focus_mover_config

from virtualunits.HAL import HAL
from virtualunits.vu_node_application import VUNodeApplication
from virtualunits.vu_measurement_unit import VUMeasurementUnit
from virtualunits.VUFocusMover import VUFocusMover
from virtualunits.VirtualTemperatureUnit import VirtualTemperatureUnit
from virtualunits.VirtualFanUnit import VUFanControl
from virtualunits.meas_seq_generator import meas_seq_generator
from virtualunits.meas_seq_generator import TriggerSignal
from virtualunits.meas_seq_generator import OutputSignal
from virtualunits.meas_seq_generator import MeasurementChannel
from virtualunits.meas_seq_generator import IntegratorMode
from virtualunits.meas_seq_generator import AnalogControlMode

from urpc.nodefunctions import NodeFunctions
from urpc.measurementfunctions import MeasurementFunctions

from urpc_enum.nodeparameter import NodeParameter
from urpc_enum.measurementparameter import MeasurementParameter

hal_unit: HAL = VUnits.instance.hal
fmb_unit: VUNodeApplication = hal_unit.nodes['Mainboard']
eef_unit: VUNodeApplication = hal_unit.nodes['EEFNode']
mc6_unit: VUNodeApplication = hal_unit.nodes['MC6']
meas_unit: VUMeasurementUnit = hal_unit.measurementUnit
usfm_unit: VUFocusMover = hal_unit.usLumFocusMover
pmt1_cooling: VirtualTemperatureUnit = hal_unit.pmt_ch1_Cooling
pmt2_cooling: VirtualTemperatureUnit = hal_unit.pmt_ch2_Cooling
hts_alpha_cooling: VirtualTemperatureUnit = hal_unit.alphaLaserHTS_Cooling
uslum_fan: VUFanControl = hal_unit.usLum_Fan
base_fan: VUFanControl = hal_unit.powerCompartment_Fan

fmb_endpoint: NodeFunctions = fmb_unit.endpoint
eef_endpoint: NodeFunctions = eef_unit.endpoint
mc6_endpoint: NodeFunctions = mc6_unit.endpoint
meas_endpoint: MeasurementFunctions = meas_unit.endpoint

report_path = hal_unit.get_config(hal_config.Application.GCReportPath)
images_path = os.path.join(report_path, 'pmt_adjust_images')
archive_path = os.path.join(report_path, 'pmt_adjust_archive')
preliminary_path = os.path.join(report_path, 'pmt_adjust_preliminary')
calibration_path = os.path.join(report_path, 'pmt_adjust_calibration')

os.makedirs(report_path, exist_ok=True)
os.makedirs(images_path, exist_ok=True)
os.makedirs(archive_path, exist_ok=True)
os.makedirs(preliminary_path, exist_ok=True)
os.makedirs(calibration_path, exist_ok=True)

config_file = os.path.join(calibration_path, 'pmt_adjust_config.json')
if os.path.isfile(config_file):
    with open(config_file, 'r') as file:
        config = json.load(file)
else:
    config = {}

if "pmt_signal_scan_default" not in config:
    config["pmt_signal_scan_default"] = {
        "led_current_dimmed": [1,2,3,4,5,7,10,13,18,24,33,45,63,88,124],    # [0:15]
        "led_current_bright": [2,3,4,5,7,10,13,18,25,33,46,64,90,127],      # [15:29]
    }
if "pmt_signal_scan_cal_hv_scan" not in config:
    config["pmt_signal_scan_cal_hv_scan"] = {
        "led_current_dimmed": [],
        "led_current_bright": [1,2,3,4,5,7,10,13,18,25,33,46,64,90,127],    # [0:15]
    }
if "pmt_signal_scan_dim_sample_scan" not in config:
    config["pmt_signal_scan_dim_sample_scan"] = {
        "led_current_dimmed": [1,2,3,4,5,7,10,13,18,24,33,45,63,88,124],    # [0:15]
        "led_current_bright": [],
    }

led_bright = {'pmt1':'fmb_led1_green', 'pmt2':'fmb_led3_green', 'uslum':'fmb_led1_green', 'htsal':'fmb_led1_green'}
led_dimmed = {'pmt1':'fmb_led2_green', 'pmt2':'fmb_led4_green', 'uslum':'fmb_led2_green', 'htsal':'fmb_led2_green'}

pmt_set_dl_delay = 0.1
pmt_set_hv_delay = 0.2
pmt_set_hv_enable_delay = 1.0   # ~600ms settling time were measured


async def fbd_pmt_adjustment(identification=''):
    """
    FBD-PMT adjustment of "Discriminator Level", "High Voltage Setting", "Pulse Pair Resolution" and "Analog Scalings".
    This adjustment only works on the test bench and not in an instrument.

    Args:
        identification:
            An identification used for the report file name. Leave empty to use the current timestamp instead.
            Allowed characters are [A-Z0-9_-] and lower case characters will be capitalized.
            Only reports of complete adjustments are saved in the archive 
            and existing reports with the same identification are overwritten.
    """
    channel = 'pmt1;pmt2'
    await pmt_adjustment(channel, identification)

    if GlobalVar.get_stop_gc():
        return f"fbd_pmt_adjustment stopped by user"


async def usl_pmt_adjustment(identification=''):
    """
    US-LUM PMT adjustment of "Discriminator Level", "High Voltage Setting" and "Pulse Pair Resolution".
    This adjustment only works on the test bench and not in an instrument.

    Args:
        identification:
            An identification used for the report file name. Leave empty to use the current timestamp instead.
            Allowed characters are [A-Z0-9_-] and lower case characters will be capitalized.
            Only reports of complete adjustments are saved in the archive 
            and existing reports with the same identification are overwritten.
    """
    channel = 'uslum;htsal'
    await pmt_adjustment(channel, identification)

    if GlobalVar.get_stop_gc():
        return f"usl_pmt_adjustment stopped by user"


async def pmt_adjustment(channel='pmt1;pmt2;uslum;htsal', identification='', scan_type='default'):

    timestamp = datetime.now()
    
    channel = str(channel).lower() if (channel != '') else 'pmt1'
    identification = str(identification).upper() if (identification != '') else timestamp.strftime('%Y%m%d_%H%M%S')
    scan_type = str(scan_type) if (scan_type != '') else 'default'

    channel = parse_channel_parameter(channel, mask=('pmt1','pmt2','uslum','htsal'))

    if not channel:
        raise ValueError(f"channel must at least contain 'pmt1', 'pmt2', 'uslum' or 'htsal'.")
    if not re.match('[A-Z0-9_-]+', identification):
        raise ValueError(f"identification must only contain [A-Z0-9_-] characters.")

    for ch in channel:
        if (f"{ch}_pdd_scaling" not in config) or (f"{ch}_led_scaling" not in config):
            raise Exception(f"Test bench calibration is missing")

    report_file = os.path.join(preliminary_path, f"{identification}_Adjustment.csv")
    
    GlobalVar.set_stop_gc(False)
    await send_to_gc(f"Starting PMT adjustment")

    with open(report_file, 'w') as report:
        report.write(f"ID:   {identification}\n")
        report.write(f"Date: {timestamp.strftime('%Y-%m-%d %H:%M:%S')}\n")
        report.write('\n')

    await send_to_gc(f"Starting Firmware")
    await base_tester_enable(True)
    try:
        if GlobalVar.get_stop_gc():
            return f"pmt_adjustment stopped by user"

        ### PMT Stabilization ######################################################

        temperature = 18    # stabilization temperature in °C
        duration    = 30    # stabilization duration in minutes

        if 'pmt1' in channel:
            await pmt1_cooling.InitializeDevice()
            await pmt1_cooling.set_target_temperature(temperature)
            await pmt1_cooling.enable()
        if 'pmt2' in channel:
            await pmt2_cooling.InitializeDevice()
            await pmt2_cooling.set_target_temperature(temperature)
            await pmt2_cooling.enable()
        if 'htsal' in channel:
            await hts_alpha_cooling.InitializeDevice()
            await hts_alpha_cooling.set_target_temperature(-273.15)     # Force the cooling to max workload
            await hts_alpha_cooling.disable()                           # Only enabled for the dark signal measurement
        if ('uslum' in channel) or ('htsal' in channel):
            await uslum_fan.InitializeDevice()
            await uslum_fan.enable()
            await usfm_unit.InitializeDevice()
            await usfm_unit.Home()
            await usfm_unit.GotoPosition(focus_mover_config.Positions.Max)

            await send_to_gc(f"Waiting for the hood to be opened...")
            await wait_for_interlock('open', timeout=60)
            if GlobalVar.get_stop_gc():
                return f"pmt_adjustment stopped by user"

            await send_to_gc(f"Waiting for the hood to be closed...", log=True)
            await wait_for_interlock('closed', timeout=60)
            if GlobalVar.get_stop_gc():
                return f"pmt_adjustment stopped by user"

        await send_to_gc(f"Stabilizing PMT")
            
        for minute in range(duration):
            await send_to_gc(f"{duration - minute} min remaining...")
            for second in range(60):
                await asyncio.sleep(1)
                if GlobalVar.get_stop_gc():
                    return f"pmt_adjustment stopped by user"
                
        await send_to_gc(f" ")
            
        ### Discriminator Adjustment ###############################################

        await send_to_gc(f"Adjusting Discriminator Level:")

        results = await pmt_adjust_discriminator(channel, dl_start=0.0, dl_stop=1.0, dl_step=0.001, report_file=report_file)

        if GlobalVar.get_stop_gc():
            return f"pmt_adjustment stopped by user"

        dl = results

        await send_to_gc(f" ")
        await send_to_gc(dict_to_str(dl, key_suffix='_dl', value_format='.3f'))
        await send_to_gc(f" ")

        ### High Voltage Adjustment ################################################

        await send_to_gc(f"Adjusting High Voltage Setting:")

        if ('pmt1' in channel) or ('pmt2' in channel):
            results = await pmt_adjust_high_voltage(channel, dl, hv_start=0.3, hv_stop=0.73, hv_step=0.005, scan_type=scan_type, report_file=report_file)
        else:
            results = await pmt_adjust_high_voltage(channel, dl, hv_start=0.4, hv_stop=0.82, hv_step=0.005, scan_type=scan_type, report_file=report_file)

        if GlobalVar.get_stop_gc():
            return f"pmt_adjustment stopped by user"

        hv = results['hv']          # Adjusted High Voltage Setting
        ppr_ns = results['ppr_ns']  # Adjusted Pulse Pair Resolution

        await send_to_gc(f" ")
        await send_to_gc(dict_to_str(hv, key_suffix='_hv ', value_format='5.3f'))
        await send_to_gc(dict_to_str(ppr_ns, key_suffix='_ppr', value_format='5.2f', value_suffix=' ns'))
        await send_to_gc(f" ")

        for ch in channel:
            if (scan_type == 'default') and not (1.0 <= ppr_ns[ch] <= 24.0):
                raise Exception(f"Failed to adjust Pulse Pair Resolution of {ch.upper()} (Limits: 1.0 <= ppr_ns <= 24.0)")

        ### Analog Adjustment ######################################################

        if ('pmt1' in channel) or ('pmt2' in channel):
            
            await send_to_gc(f"Adjusting Analog Scaling:")

            results = await pmt_adjust_analog(channel, dl, hv, ppr_ns, report_file=report_file)

            if GlobalVar.get_stop_gc():
                return f"pmt_adjustment stopped by user"

            als = results['als']    # Adjusted Analog Low Scaling
            ahs = results['ahs']    # Adjusted Analog High Scaling

            await send_to_gc(f" ")
            await send_to_gc(dict_to_str(als, key_suffix='_als', value_format='8.6f'))
            await send_to_gc(dict_to_str(ahs, key_suffix='_ahs', value_format='8.4f'))
            await send_to_gc(f" ")

            for ch in channel:
                if (scan_type == 'default') and not (0.1 <= als[ch] <= 0.5):
                    raise Exception(f"Failed to adjust Analog Low Scaling of {ch.upper()} (Limits: 0.1 <= als <= 0.5)")
                if not (90.0 <= ahs[ch] <= 112.0):
                    raise Exception(f"Failed to adjust Analog High Scaling of {ch.upper()} (Limits: 90.0 <= ahs <= 112.0)")

        ### Dark Signal ############################################################

        await send_to_gc(f"Measuring Dark Signal:")

        if 'htsal' in channel:
            await meas_endpoint.SetParameter(MeasurementParameter.HTSAlphaLaserPower, 1.0)
            await meas_endpoint.SetParameter(MeasurementParameter.HTSAlphaLaserEnable, 1)
            await hts_alpha_cooling.enable()

        results = await pmt_dark_signal(channel, dl, hv, report_file=report_file)

        dark_mean = results['dark_mean']
        # dark_std = results['dark_std']
        # dark_std_error = results['dark_std_error']

        await send_to_gc(f" ")
        await send_to_gc(dict_to_str(dark_mean, key_suffix='_dark_mean', value_format='.1f'))
        # await send_to_gc(dict_to_str(dark_std, key_suffix='_dark_std', value_format='.3f'))
        # await send_to_gc(dict_to_str(dark_std_error, key_suffix='_dark_std_error', value_format='.2%'))
        await send_to_gc(f" ")

        dark_mean_limit = {'pmt1':200, 'pmt2':200, 'uslum':50, 'htsal':200}
        # dark_std_limit = {ch:(np.sqrt(cps) * 1.5) for ch, cps in dark_cps_limit.items()}
        # dark_std_error_limit = 0.5

        for ch in channel:
            if dark_mean[ch] > dark_mean_limit[ch]:
                raise Exception(f"Dark signal measurement failed. {ch.upper()} exceeds the limit.")
        #     if dark_std[ch] > dark_std_limit[ch]:
        #         raise Exception(f"Dark signal measurement failed. {ch.upper()} exceeds the limit.")
        #     if dark_std_error[ch] > dark_std_error_limit[ch]:
        #         raise Exception(f"Dark signal measurement failed. {ch.upper()} exceeds the limit.")

        ### Adjustment Results #####################################################

        pmt_name = {'pmt1':'PMT1', 'pmt2':'PMT2', 'uslum':'PMT_USLUM', 'htsal':'PMT_HTSAlpha'}

        await send_to_gc(f"Adjusted Parameter:")
        for ch in channel:
            await send_to_gc(f" ")
            await send_to_gc(f"{pmt_name[ch]}")
            if ch in ['pmt1', 'pmt2']:
                await send_to_gc(f"|-----|----------|")
                await send_to_gc(f"| DL  | {dl[ch]:8.3f} |")
                await send_to_gc(f"| HV  | {hv[ch]:8.3f} |")
                await send_to_gc(f"| PPR | {ppr_ns[ch]:5.2f}e-9 |")
                await send_to_gc(f"| ACE | {als[ch]:8.6f} |")
                await send_to_gc(f"| AHS | {ahs[ch]:8.4f} |")
                await send_to_gc(f"|-----|----------|")
            if ch in ['uslum', 'htsal']:
                await send_to_gc(f"|------------|------------|--------------------|")
                await send_to_gc(f"| HV setting | DL values  | PMT puls pair res. |")
                await send_to_gc(f"|------------|------------|--------------------|")
                await send_to_gc(f"| {hv[ch]:10.3f} | {dl[ch]:10.3f} | {ppr_ns[ch]:15.2f}e-9 |")
                await send_to_gc(f"|------------|------------|--------------------|")

        with open(report_file, 'a') as report:
            report.write(f"Adjusted Parameter:\n")
            for ch in channel:
                report.write('\n')
                report.write(f"DiscriminatorLevel_{pmt_name[ch]} = {dl[ch]:.3f}\n")
                report.write(f"HighVoltageSetting_{pmt_name[ch]} = {hv[ch]:.3f}\n")
                report.write(f"Pulse_pair_res_{pmt_name[ch]}_s = {ppr_ns[ch]:.2f}e-9\n")
                if ch in ('pmt1', 'pmt2'):
                    report.write(f"AnalogCountingEquivalent_{pmt_name[ch]} = {als[ch]:.6f}\n")
                    report.write(f"AnalogHighRangeScale_{pmt_name[ch]} = {ahs[ch]:.3f}\n")

        shutil.copy(report_file, os.path.join(archive_path, f"{identification}_Adjustment.csv"))
        os.remove(report_file)

    finally:
        if 'htsal' in channel:
            await meas_endpoint.SetParameter(MeasurementParameter.HTSAlphaLaserEnable, 0)
            await hts_alpha_cooling.disable()
        if ('uslum' in channel) or ('htsal' in channel):
            await usfm_unit.Move(0)
        await base_tester_enable(False)


async def pmt_adjust_calibration(channel='pmt1;pmt2;uslum;htsal'):

    timestamp = datetime.now()
    
    channel = str(channel).lower() if (channel != '') else 'pmt1'
    channel = parse_channel_parameter(channel, mask=('pmt1','pmt2','uslum','htsal'))

    if not channel:
        raise ValueError(f"channel must at least contain 'pmt1', 'pmt2', 'uslum' or 'htsal'.")

    report_file = os.path.join(calibration_path, f"pmt_adjust_calibration_{timestamp.strftime('%Y%m%d_%H%M%S')}.csv")
    
    GlobalVar.set_stop_gc(False)
    await send_to_gc(f"Starting PMT test bench calibration")

    with open(report_file, 'a') as report:
        report.write(f"Old Calibration:\n")
        report.write(json.dumps(config, indent=4))
        report.write('\n')
        report.write('\n')
        
    config['calibration_date'] = timestamp.strftime('%Y-%m-%d %H:%M:%S')

    await send_to_gc(f"Starting Firmware")
    await base_tester_enable(True)
    try:
        if GlobalVar.get_stop_gc():
            return f"pmt_adjust_calibration stopped by user"

        ### PMT Stabilization ######################################################

        temperature = 18    # stabilization temperature in °C
        duration    = 30    # stabilization duration in minutes

        if 'pmt1' in channel:
            await pmt1_cooling.InitializeDevice()
            await pmt1_cooling.set_target_temperature(temperature)
            await pmt1_cooling.enable()
        if 'pmt2' in channel:
            await pmt2_cooling.InitializeDevice()
            await pmt2_cooling.set_target_temperature(temperature)
            await pmt2_cooling.enable()
        if ('uslum' in channel) or ('htsal' in channel):
            await uslum_fan.InitializeDevice()
            await uslum_fan.enable()
            await usfm_unit.InitializeDevice()
            await usfm_unit.Home()
            await usfm_unit.GotoPosition(focus_mover_config.Positions.Max)

            await send_to_gc(f"Waiting for the hood to be opened...")
            await wait_for_interlock('open', timeout=60)
            if GlobalVar.get_stop_gc():
                return f"pmt_adjust_calibration stopped by user"

            await send_to_gc(f"Waiting for the hood to be closed...", log=True)
            await wait_for_interlock('closed', timeout=60)
            if GlobalVar.get_stop_gc():
                return f"pmt_adjust_calibration stopped by user"

        await send_to_gc(f"Stabilizing PMT")

        for minute in range(duration):
            await send_to_gc(f"{duration - minute} min remaining...")
            for second in range(60):
                await asyncio.sleep(1)
                if GlobalVar.get_stop_gc():
                    return f"pmt_adjustment stopped by user"

        await send_to_gc(f" ")

        ### Photodiode Analog Scaling ##############################################

        await send_to_gc(f"Calibrating Photodiode Analog Scaling:")

        results = await pdd_adjust_analog(channel, report_file=report_file)

        if GlobalVar.get_stop_gc():
            return f"pmt_adjust_calibration stopped by user"
        
        for ch in channel:
            config[f"{ch}_pdd_scaling"] = results[ch]

        await send_to_gc(f" ")
        await send_to_gc(f"pdd_scaling = {results}")
        await send_to_gc(f" ")

        ### High Voltage Adjustment ################################################

        await send_to_gc(f"Adjusting PMT Settings for the Calibration:")

        dl = {'pmt1':0.3, 'pmt2':0.3, 'uslum':0.45, 'htsal':0.45}

        results = await pmt_adjust_high_voltage(channel, dl, hv_start=0.3, hv_stop=0.7, hv_step=0.005, scan_type='cal_hv_scan', report_file=report_file)

        if GlobalVar.get_stop_gc():
            return f"pmt_adjust_calibration stopped by user"

        hv = results['hv']          # Adjusted High Voltage Setting
        ppr_ns = results['ppr_ns']  # Adjusted Pulse Pair Resolution

        await send_to_gc(f" ")
        await send_to_gc(f"hv = {hv}, ppr_ns = {ppr_ns}")
        await send_to_gc(f" ")

        ### Photodiode LED Scaling #################################################

        await send_to_gc(f"Calibrating Photodiode LED Scaling:")

        results = await pdd_adjust_led_scaling(channel, dl, hv, ppr_ns, report_file=report_file)

        if GlobalVar.get_stop_gc():
            return f"pmt_adjust_calibration stopped by user"
        
        for ch in channel:
            config[f"{ch}_led_scaling"] = results[ch]

        await send_to_gc(f" ")
        await send_to_gc(f"led_scaling = {results}")
        await send_to_gc(f" ")

        with open(report_file, 'a') as report:
            report.write(f"New Calibration:\n")
            report.write(json.dumps(config, indent=4))
            report.write('\n')

        with open(config_file, 'w') as file:
            json.dump(config, file, indent=4)

    finally:
        if ('uslum' in channel) or ('htsal' in channel):
            await usfm_unit.Move(0)
        await base_tester_enable(False)


async def pmt_adjust_discriminator(channel, dl_start, dl_stop, dl_step, report_file=''):

    channel = parse_channel_parameter(channel, mask=('pmt1','pmt2','uslum','htsal'))

    window_ms = 100.0
    window_count = 1
    iterations = 1

    dl_offset = {'pmt1':0.05, 'pmt2':0.05, 'uslum':0.075, 'htsal':0.075}    # DL is set at a fixed offset to the noise peak of the scan.
    dl_width_limit = {'pmt1':0.04, 'pmt2':0.04, 'uslum':0.06, 'htsal':0.06} # The width of the noise peak must not be larger than this limit.

    report_file = str(report_file) if (report_file != '') else os.path.join(report_path, f"pmt_adjust_discriminator.csv")

    await base_tester_enable(True)
    try:
        with open(report_file, 'a') as report:
            timestamp = datetime.now().strftime('%Y%m%d_%H%M%S')
            report.write(f"pmt_adjust_discriminator(channel={channel}, dl_start={dl_start:.3f}, dl_stop={dl_stop:.3f}, dl_step={dl_step:.3f}) started at {timestamp}\n")
            report.write(f"temperature: {await pmt_get_temperature(channel)}\n")
            report.write('\n')
            report.write(f"parameter: window_ms={window_ms}, window_count={window_count}, iterations={iterations}, dl_offset={dl_offset}, dl_width_limit={dl_width_limit}\n")
            report.write('\n')
        
            cps = {}    # Counts Per Second (The measured noise)
            dt = {}     # Dead Time (The amount of time the PMT signal was above the discriminator level)
            
            dl_range = np.arange(dl_start, (dl_stop + 1e-6), dl_step).round(6)  # The discriminator level scan range

            output = f"dl    ; "
            for ch in channel:
                cps[ch] = np.zeros_like(dl_range)
                dt[ch] = np.zeros_like(dl_range)

                output += f"{ch + '_cps':10} ; {ch + '_dt':10} ; "

            await send_to_gc(output, report=report)

            await pmt_set_hv_enable(channel, 0)

            for i, dl in enumerate(dl_range):
                await pmt_set_dl(channel, dl)
                await asyncio.sleep(pmt_set_dl_delay)
                results = await pmt_counting_measurement(window_ms, window_count, iterations)
                if GlobalVar.get_stop_gc():
                    return f"pmt_adjust_discriminator stopped by user"

                output = f"{dl:5.3f} ; "
                for ch in channel:
                    cps[ch][i] = results[f"{ch}_cps_mean"]
                    dt[ch][i] = results[f"{ch}_dt_mean"]

                    output += f"{cps[ch][i]:10.0f} ; {dt[ch][i]:10.0f} ; "

                await send_to_gc(output, report=report)

            report.write('\n')

            plot_min = 1.0
            plot_max = 0.0

            dl = {}

            for ch in channel:
                cps_max = np.max(cps[ch])
                dt_max = np.max(dt[ch])

                dl_peak = dl_range[cps[ch] > (cps_max * 0.9)]
                dt_peak = dl_range[dt[ch] > (dt_max * 0.5)]

                if (len(dl_peak) > 0) and (cps_max >= 100):
                    dl_center = (np.max(dl_peak) + np.min(dl_peak)) / 2
                elif (len(dt_peak > 0)):
                    dl_center = np.max(dt_peak)
                else:
                    raise Exception(f"Failed to adjust discriminator level. No peak found in the measurement.")

                dl[ch] = np.round((dl_center + dl_offset[ch]), 3)

                plot_min = min(plot_min, (dl_center - dl_offset[ch]))
                plot_max = max(plot_max, (dl_center + 2 * dl_offset[ch]))

                #TODO Filter outlier counts when zero before and after.
                dl_noise = dl_range[cps[ch] > 0.0]
                dl_width = np.max(dl_noise) - dl_center if (len(dl_noise) > 0) else 0.0

                if dl_width > dl_width_limit[ch]:
                    raise Exception(f"Failed to adjust discriminator level. Peak is too wide. (center: {dl_center:.3f}, width: {dl_width:.3f})")

            report.write(dict_to_str(dl, key_suffix='_dl', value_format='.3f', value_separator=' ; ', item_separator=' ; '))
            report.write('\n')
            report.write('\n')

            # Plot only the relevant part of the scan.
            plot_select = np.logical_and((dl_range >= plot_min), (dl_range <= plot_max))
            dl_range = dl_range[plot_select]
            for ch in channel:
                cps[ch] = cps[ch][plot_select]
                dt[ch] = dt[ch][plot_select]
            
            await plot_dl_scan(channel, dl_range, cps, dt, dl, file_name='dl_scan.png')

        return dl

    finally:
        await base_tester_enable(False)


async def pmt_adjust_high_voltage(channel, dl, hv_start, hv_stop, hv_step, scan_type='default', report_file=''):

    channel = parse_channel_parameter(channel, mask=('pmt1','pmt2','uslum','htsal'))

    scan_type = str(scan_type) if (scan_type != '') else 'default'

    linearity_limit = 0.9997    # Select the highest high voltage setting with an r² value greater or equal to this limit.
    linearity_stop  = 0.997     # Stop the scan when linearity drops below this limit.

    rel_error_limit = 0.10      # Select the highest high voltage setting with an relative error less or equal to this limit.
    rel_error_stop  = 0.25      # Stop the scan when relative error rises above this limit.

    rel_gain = {'pmt1':None, 'pmt2':None, 'uslum':None, 'htsal':None}

    if scan_type == 'default':
        window_ms = 50
        window_count = 20
        cps_min = 1000
        limit_criterion = 'rel_error_limit'         # HV must be within the relative error limit.
        selection_criterion = 'gain_max'            # Select the highest gain.
        # if 'uslum' in channel:
        #     rel_gain['htsal'] = {'channel':'uslum', 'scaling':1.8, 'tolerance':0.05}
    if scan_type == 'cal_hv_scan':
        window_ms = 50
        window_count = 20
        cps_min = 1000
        limit_criterion = 'linearity_limit'         # HV must be within the linearity limit.
        selection_criterion = 'linearity_max'       # Select the highest linearity.
    if scan_type == 'dim_sample_scan':
        window_ms = 100
        window_count = 1
        cps_min = 1000
        limit_criterion = 'rel_error_limit'                 # HV must be within the relative error limit.
        selection_criterion = 'sensitivity_filtered_min'    # Select the best sensitivity.

    report_file = str(report_file) if (report_file != '') else os.path.join(report_path, f"pmt_adjust_high_voltage.csv")

    timestamp = datetime.now().strftime('%Y%m%d_%H%M%S')
    
    ppr_ns = {}         # The pulse pair resolution in ns.
    sensitivity = {}    # The measured sensitivity.
    gain = {}           # The signal gain of the PMT.
    linearity = {}      # The measured linearity (r² value).
    rel_error = {}      # The measured relative error.

    await base_tester_enable(True)
    try:
        hv_start = await pmt_get_hv_start(channel, dl, hv_start, hv_stop, hv_step, cps_min)
        if GlobalVar.get_stop_gc():
            return f"pmt_adjust_high_voltage stopped by user"
        
        hv_range = np.arange(hv_start, (hv_stop + 1e-6), hv_step).round(6)  # The high voltage setting scan range.

        output = f"hv    ; "
        for ch in channel:
            ppr_ns[ch] = np.zeros_like(hv_range)
            sensitivity[ch] = np.zeros_like(hv_range)
            gain[ch] = np.zeros_like(hv_range)
            linearity[ch] = np.zeros_like(hv_range)
            rel_error[ch] = np.ones_like(hv_range)      # Assuming rel_error_stop < 1.0

            output += f"{ch + '_ppr':10} ; {ch + '_sen':10} ; {ch + '_gain':10} ; {ch + '_lin':10} ; {ch + '_err':10} ; "

        await send_to_gc(output)

        for i, hv in enumerate(hv_range):
            results = await pmt_signal_scan(channel, dl, hv, window_ms, window_count, scan_type, report_file=report_file)
            if GlobalVar.get_stop_gc():
                return f"pmt_adjust_high_voltage stopped by user"

            continue_scan = False

            output = f"{hv:5.3f} ; "
            for ch in channel:
                ppr_ns[ch][i] = results['ppr_ns'][ch]
                sensitivity[ch][i] = results['sensitivity'][ch]
                gain[ch][i] = results['gain'][ch]
                linearity[ch][i] = results['linearity'][ch]
                rel_error[ch][i] = results['rel_error'][ch]

                output += f"{ppr_ns[ch][i]:7.2f}e-9 ; {sensitivity[ch][i]:10.3f} ; {gain[ch][i]:10.3f} ; {linearity[ch][i]:10.6f} ; {rel_error[ch][i]:10.2%} ; "

                if (limit_criterion == 'linearity_limit') and ((linearity[ch][i] >= linearity_stop) or (np.max(linearity[ch]) < linearity_limit)):
                    continue_scan = True
                if (limit_criterion == 'rel_error_limit') and ((rel_error[ch][i] <= rel_error_stop) or (np.min(rel_error[ch]) > rel_error_limit)):
                    continue_scan = True

            await send_to_gc(output)

            if not continue_scan:
                hv_range = hv_range[:(i + 1)]
                for ch in channel:
                    ppr_ns[ch] = ppr_ns[ch][:(i + 1)]
                    sensitivity[ch] = sensitivity[ch][:(i + 1)]
                    gain[ch] = gain[ch][:(i + 1)]
                    linearity[ch] = linearity[ch][:(i + 1)]
                    rel_error[ch] = rel_error[ch][:(i + 1)]
                break

        with open(report_file, 'a') as report:
            report.write(f"pmt_adjust_high_voltage(channel={channel}, dl={dl}, hv_start={hv_start:.3f}, hv_stop={hv_stop:.3f}, hv_step={hv_step:.3f}, scan_type={scan_type}) started at {timestamp}\n")
            report.write('\n')
            report.write(f"parameter: window_ms={window_ms}, window_count={window_count}, cps_min={cps_min}, limit_criterion={limit_criterion}, selection_criterion={selection_criterion}, rel_gain={rel_gain}, linearity_limit={linearity_limit}, linearity_stop={linearity_stop}, rel_error_limit={rel_error_limit}, rel_error_stop={rel_error_stop}\n")
            report.write('\n')

            report.write(f"hv    ; ")
            for ch in channel:
                report.write(f"{ch + '_ppr':10} ; {ch + '_sen':10} ; {ch + '_gain':10} ; {ch + '_lin':10} ; {ch + '_err':10} ; ")
            report.write('\n')

            for i, hv in enumerate(hv_range):
                report.write(f"{hv:5.3f} ; ")
                for ch in channel:
                    report.write(f"{ppr_ns[ch][i]:7.2f}e-9 ; {sensitivity[ch][i]:10.3f} ; {gain[ch][i]:10.3f} ; {linearity[ch][i]:10.6f} ; {rel_error[ch][i]:10.2%} ; ")
                report.write('\n')

            report.write('\n')

            hv = {}

            # Data slices after limit criterion
            hv_range_slice = {}
            ppr_ns_slice = {}
            sensitivity_slice = {}
            gain_slice = {}
            linearity_slice = {}
            rel_error_slice = {}

            for ch in channel:
                if not limit_criterion:
                    linear_range = np.full_like(hv_range, True)
                elif limit_criterion == 'linearity_limit':
                    linear_range = linearity[ch] >= linearity_limit
                elif limit_criterion == 'rel_error_limit':
                    linear_range = rel_error[ch] <= rel_error_limit
                else:
                    raise Exception(f"Invalid limit criterion")

                if not np.any(linear_range):
                    raise Exception(f"Failed to adjust high voltage setting of {ch.upper()}. Required linearity not reached.")

                hv_range_slice[ch] = hv_range[linear_range]
                ppr_ns_slice[ch] = ppr_ns[ch][linear_range]
                sensitivity_slice[ch] = sensitivity[ch][linear_range]
                gain_slice[ch] = gain[ch][linear_range]
                linearity_slice[ch] = linearity[ch][linear_range]
                rel_error_slice[ch] = rel_error[ch][linear_range]

                if selection_criterion == 'hv_max':
                    hv[ch] = np.max(hv_range_slice[ch])
                elif selection_criterion == 'linearity_max':
                    hv[ch] = hv_range_slice[ch][linearity_slice[ch] == np.max(linearity_slice[ch])][0]
                elif selection_criterion == 'rel_error_min':
                    hv[ch] = hv_range_slice[ch][rel_error_slice[ch] == np.min(rel_error_slice[ch])][0]
                elif selection_criterion == 'gain_max':
                    hv[ch] = hv_range_slice[ch][gain_slice[ch] == np.max(gain_slice[ch])][0]
                elif selection_criterion == 'sensitivity_filtered_min':
                    sensitivity_filtered = binomial_filter(sensitivity_slice[ch], iterations=15)
                    hv[ch] = hv_range_slice[ch][sensitivity_filtered == np.min(sensitivity_filtered)][0]

            for ch in channel:
                #TODO USL gain can have a local maximum and the adjustment probably shouldn't pick a hv setting above the local maximum.
                if rel_gain[ch]:
                    await send_to_gc(f"rel_gain[{ch}]: {rel_gain[ch]}")
                    ref_channel = rel_gain[ch]['channel']
                    gain_scaling = rel_gain[ch]['scaling']
                    gain_tolerance = rel_gain[ch]['tolerance']
                    gain_target = gain_slice[ref_channel][hv_range_slice[ref_channel] == hv[ref_channel]][0] * gain_scaling
                    await send_to_gc(f"gain_target = {gain_target:.3f}")
                    hv[ch] = hv_range_slice[ch][np.abs(gain_slice[ch] - gain_target).argmin()]
                    if hv[ch] not in hv_range_slice[ch]:
                        raise Exception(f"Failed to adjust high voltage setting of {ch.upper()}. Required linearity not reached for targeted gain.")
                    gain_error = np.abs(gain_slice[ch][hv_range_slice[ch] == hv[ch]][0] - gain_target) / gain_target
                    await send_to_gc(f"gain_value  = {gain_slice[ch][hv_range_slice[ch] == hv[ch]][0]:.3f}")
                    await send_to_gc(f"gain_error  = {gain_error:.2%}")
                    if gain_error > gain_tolerance:
                        raise Exception(f"Failed to adjust high voltage setting of {ch.upper()}. Gain is outside of targeted gain tolerance.")

            for ch in channel:
                ppr_ns[ch] = ppr_ns_slice[ch][hv_range_slice[ch] == hv[ch]][0]

            report.write(dict_to_str(hv, key_suffix='_hv', value_format='.3f', value_separator=' ; ', item_separator=' ; '))
            report.write('\n')
            report.write(dict_to_str(ppr_ns, key_suffix='_ppr', value_format='.2f', value_suffix='e-9', value_separator=' ; ', item_separator=' ; '))
            report.write('\n')
            report.write('\n')

            await plot_hv_scan(channel, hv_range, gain, hv, title='HV Gain Scan', file_name='hv_gain_scan.png')
            if scan_type == 'dim_sample_scan':
                await plot_hv_scan(channel, hv_range, sensitivity, hv, title='HV Sensitivity Scan', file_name='hv_sensitivity_scan.png')

        return {'hv':hv, 'ppr_ns':ppr_ns}
    
    finally:
        await base_tester_enable(False)


async def pmt_get_hv_start(channel, dl, hv_start=0.3, hv_stop=0.7, hv_step=0.005, cps_min=1000):

    channel = parse_channel_parameter(channel, mask=('pmt1','pmt2','uslum','htsal'))

    for led in np.unique(list(led_dimmed.values()) + list(led_bright.values())):
        led_source, led_channel, led_type = led.split('_')
        await set_led_current(0, led_source, led_channel, led_type)

    try:
        await pmt_set_dl(channel, dl)
        await pmt_set_hv(channel, hv_start)
        await asyncio.sleep(pmt_set_hv_delay)
        await pmt_set_hv_enable(channel, 1)
        await asyncio.sleep(pmt_set_hv_enable_delay)

        for ch in channel:
            led_source, led_channel, led_type = led_dimmed[ch].split('_')
            await set_led_current(1, led_source, led_channel, led_type)

        hv_range = np.arange(hv_start, (hv_stop + 1e-6), hv_step).round(6)

        for hv in hv_range:
            await pmt_set_hv(channel, hv)
            await asyncio.sleep(0.5)
            results = await pmt_counting_measurement(window_ms=100, window_count=1, iterations=1)
            if GlobalVar.get_stop_gc():
                return f"pmt_get_hv_start stopped by user"
            
            for ch in channel:
                if results[f"{ch}_cps_mean"] >= cps_min:
                    return round(hv, 6)

            output = f"hv={hv:.3f} skipped ("
            for ch in channel:
                output += f"{ch}={results[ch + '_cps_mean']}, "

            await send_to_gc(output[:-2] + ')', log=True)

        raise Exception(f"The count rate is below the limit for all high voltage settings.")

    finally:
        await pmt_set_hv_enable(channel, 0)


async def pmt_signal_scan(channel, dl, hv, window_ms=50, window_count=20, scan_type='default', report_file=''):

    channel = parse_channel_parameter(channel, mask=('pmt1','pmt2','uslum','htsal'))
    
    scan_type = str(scan_type) if (scan_type != '') else 'default'

    scan_config = config[f"pmt_signal_scan_{scan_type}"]

    led_current_dimmed = scan_config['led_current_dimmed']
    led_current_bright = scan_config['led_current_bright']

    scan_size = len(led_current_dimmed) + len(led_current_bright)

    if scan_type == 'default':
        dark_iterations = 0
        dark_window_ms = window_ms
        dark_window_count = window_count
        lin_ppr_size = {'pmt1':27, 'pmt2':27, 'uslum':27, 'htsal':25}
        lin_reg_size = 15
        sen_ref_index = 14
        linearization = 'r_squared_min'
    if scan_type == 'cal_hv_scan':
        dark_iterations = 0
        dark_window_ms = window_ms
        dark_window_count = window_count
        lin_ppr_size = {'pmt1':13, 'pmt2':13, 'uslum':13, 'htsal':13}
        lin_reg_size = 2
        sen_ref_index = 0
        linearization = 'r_squared_min'
    if scan_type == 'dim_sample_scan':
        dark_iterations = 30
        dark_window_ms = 50
        dark_window_count = 20
        lin_ppr_size = {'pmt1':15, 'pmt2':15, 'uslum':15, 'htsal':15}
        lin_reg_size = 7
        sen_ref_index = 14
        linearization = 'r_squared_min'

    dark_cps = {}       # The mean value of the dark measurement in counts per second.
    dark_std = {}       # The standard deviation of the dark measurement in counts per second.
    signal_cps = {}     # The measured signal values of the pmt in counts per second.
    signal_ref = {}     # The measured signal values of the photodiode.
    
    report_file = str(report_file) if (report_file != '') else os.path.join(report_path, f"pmt_signal_scan.csv")

    await base_tester_enable(True)
    try:
        with open(report_file, 'a') as report:
            timestamp = datetime.now().strftime('%Y%m%d_%H%M%S')
            report.write(f"pmt_signal_scan(channel={channel}, dl={dl}, hv={hv}, window_ms={window_ms}, window_count={window_count}, scan_type={scan_type}) started at {timestamp}\n")
            report.write(f"temperature: {await pmt_get_temperature(channel)}\n")
            report.write('\n')
            report.write(f"parameter: dark_iterations={dark_iterations}, dark_window_ms={dark_window_ms}, dark_window_count={dark_window_count}, led_current_dimmed={led_current_dimmed}, led_current_bright={led_current_bright}, lin_ppr_size={lin_ppr_size}, lin_reg_size={lin_reg_size}, sen_ref_index={sen_ref_index}, linearization={linearization}\n")
            report.write('\n')

            for led in np.unique(list(led_dimmed.values()) + list(led_bright.values())):
                led_source, led_channel, led_type = led.split('_')
                await set_led_current(0, led_source, led_channel, led_type)

            await pmt_set_dl(channel, dl)
            await pmt_set_hv(channel, hv)
            await asyncio.sleep(pmt_set_hv_delay)
            await pmt_set_hv_enable(channel, 1)
            await asyncio.sleep(pmt_set_hv_enable_delay)

            if dark_iterations > 0:
                results = await pmt_counting_measurement(dark_window_ms, dark_window_count, dark_iterations)
                if GlobalVar.get_stop_gc():
                    return f"pmt_signal_scan stopped by user"

                for ch in channel:
                    dark_cps[ch] = results[f"{ch}_cps_mean"]
                    dark_std[ch] = results[f"{ch}_cps_std"]

                report.write(dict_to_str(dark_cps, key_suffix='_dark_cps', value_format='.6f', value_separator=' ; ', item_separator=' ; '))
                report.write('\n')
                report.write(dict_to_str(dark_std, key_suffix='_dark_std', value_format='.6f', value_separator=' ; ', item_separator=' ; '))
                report.write('\n')
                report.write('\n')

            for ch in channel:
                signal_cps[ch] = np.zeros(scan_size)
                signal_ref[ch] = np.zeros(scan_size)

                report.write(f"{ch + '_signal':12} ; {ch + '_cps':12} ; ")

            report.write('\n')

            ppr_ns = {ch:0.0 for ch in channel}
            sensitivity = {ch:0.0 for ch in channel}
            gain = {ch:0.0 for ch in channel}
            linearity = {ch:0.0 for ch in channel}
            rel_error = {ch:1.0 for ch in channel}

            index = 0

            for i, _ in enumerate(led_current_dimmed):
                for ch in channel:
                    normalization = scan_config.get(f"{ch}_normalization", {}).get(f"led_current_dimmed", None)
                    current = led_current_dimmed[i] if not normalization else normalization[i]
                    led_source, led_channel, led_type = led_dimmed[ch].split('_')
                    await set_led_current(current, led_source, led_channel, led_type)

                results = await pmt_counting_measurement(window_ms, window_count, iterations=1)
                if GlobalVar.get_stop_gc():
                    return f"pmt_signal_scan stopped by user"

                for ch in channel:
                    signal_cps[ch][index] = results[f"{ch}_cps_mean"]
                    signal_ref[ch][index] = results[f"{ch}_pdd_mean"] * config[f"{ch}_led_scaling"]

                    report.write(f"{signal_ref[ch][index]:12.0f} ; {signal_cps[ch][index]:12.0f} ; ")

                report.write('\n')
                index += 1
            
            for ch in channel:
                led_source, led_channel, led_type = led_dimmed[ch].split('_')
                await set_led_current(0, led_source, led_channel, led_type)

            for i, _ in enumerate(led_current_bright):
                for ch in channel:
                    normalization = scan_config.get(f"{ch}_normalization", {}).get(f"led_current_bright", None)
                    current = led_current_bright[i] if not normalization else normalization[i]
                    led_source, led_channel, led_type = led_bright[ch].split('_')
                    await set_led_current(current, led_source, led_channel, led_type)

                results = await pmt_counting_measurement(window_ms, window_count, iterations=1)
                if GlobalVar.get_stop_gc():
                    return f"pmt_signal_scan stopped by user"

                for ch in channel:
                    signal_cps[ch][index] = results[f"{ch}_cps_mean"]
                    signal_ref[ch][index] = results[f"{ch}_pdd_mean"]

                    report.write(f"{signal_ref[ch][index]:12.0f} ; {signal_cps[ch][index]:12.0f} ; ")

                report.write('\n')
                index += 1
            
            for ch in channel:
                led_source, led_channel, led_type = led_bright[ch].split('_')
                await set_led_current(0, led_source, led_channel, led_type)

            report.write('\n')

            for ch in channel:
                signal_ref[ch] = signal_ref[ch][:lin_ppr_size[ch]]
                signal_cps[ch] = signal_cps[ch][:lin_ppr_size[ch]]

                ppr_ns[ch], linearity[ch], rel_error[ch] = pmt_calculate_ppr_ns(signal_ref[ch], signal_cps[ch], optimization=linearization)
                
                signal_cps[ch] = pmt_calculate_correction(signal_cps[ch], ppr_ns[ch])

                if (dark_iterations > 0) and (dark_cps[ch] < signal_cps[ch][sen_ref_index]):
                    sensitivity[ch] = 3 * signal_ref[ch][sen_ref_index] * dark_std[ch] / (signal_cps[ch][sen_ref_index] - dark_cps[ch])
                else:
                    sensitivity[ch] = 0.0
                
                gain[ch] = np.mean(signal_cps[ch][:lin_reg_size] / signal_ref[ch][:lin_reg_size])

                signal_ref[ch] = np.mean(signal_cps[ch][:lin_reg_size] / signal_ref[ch][:lin_reg_size]) * signal_ref[ch]

            report.write(dict_to_str(ppr_ns, key_suffix='_ppr', value_format='.2f', value_suffix='e-9', value_separator=' ; ', item_separator=' ; '))
            report.write('\n')
            report.write(dict_to_str(sensitivity, key_suffix='_sen', value_format='.3f', value_separator=' ; ', item_separator=' ; '))
            report.write('\n')
            report.write(dict_to_str(gain, key_suffix='_gain', value_format='.3f', value_separator=' ; ', item_separator=' ; '))
            report.write('\n')
            report.write(dict_to_str(linearity, key_suffix='_lin', value_format='.6f', value_separator=' ; ', item_separator=' ; '))
            report.write('\n')
            report.write(dict_to_str(rel_error, key_suffix='_err', value_format='.2%', value_separator=' ; ', item_separator=' ; '))
            report.write('\n')
            report.write('\n')

            await plot_signal_scan(channel, signal_ref, signal_cps, signal_ref, hv, file_name=f"signal_scan_hv_{hv:.3f}.png")

        return {'ppr_ns':ppr_ns, 'sensitivity':sensitivity, 'gain':gain, 'linearity':linearity, 'rel_error':rel_error}
        
    finally:
        await pmt_set_hv_enable(channel, 0)
        await base_tester_enable(False)


async def pmt_adjust_analog(channel, dl, hv, ppr_ns, report_file=''):

    channel = parse_channel_parameter(channel, mask=('pmt1','pmt2'))
    
    iterations = 100

    led_current = np.linspace(1, 127, 127)
    
    al_lower_limit = round((65536 - 1310) * 0.2)    # The analog low value must be greater or equal to this limit. 20% of the analog low range with ~100 mV offset voltage.
    al_upper_limit = round((65536 - 1310) * 0.8)    # The analog low value must be less or equal to this limit. 80% of the analog low range with ~100 mV offset voltage.
    
    report_file = str(report_file) if (report_file != '') else os.path.join(report_path, f"pmt_adjust_analog.csv")

    await base_tester_enable(True)
    try:
        with open(report_file, 'a') as report:
            timestamp = datetime.now().strftime('%Y%m%d_%H%M%S')
            report.write(f"pmt_adjust_analog(channel={channel}, dl={dl}, hv={hv}, ppr_ns={ppr_ns}) started at {timestamp}\n")
            report.write(f"temperature: {await pmt_get_temperature(channel)}\n")
            report.write('\n')
            report.write(f"parameter: iterations={iterations}, al_lower_limit={al_lower_limit}, al_upper_limit={al_upper_limit}\n")
            report.write('\n')

            await pmt_set_dl(channel, dl)
            await pmt_set_hv(channel, hv)
            await asyncio.sleep(pmt_set_hv_delay)
            await pmt_set_hv_enable(channel, 1)
            await asyncio.sleep(pmt_set_hv_enable_delay)
            
            # Determine the required measurement window time in ms.
            window_ms = 1.0
            for ch in channel:
                led_source, led_channel, led_type = led_dimmed[ch].split('_')
                await set_led_current(np.max(led_current), led_source, led_channel, led_type)

            results = await pmt_multi_range_measurement(window_ms, iterations)

            al_max = 0.0
            for ch in channel:
                al_max = max(al_max, results[f"{ch}_al_mean"])
            if (al_max > al_upper_limit):
                window_ms = 0.1
                results = await pmt_multi_range_measurement(window_ms, iterations)
            
            al_max = 0.0
            for ch in channel:
                al_max = max(al_max, results[f"{ch}_al_mean"])
            if (al_max > al_upper_limit):
                raise Exception(f"Failed to adjust analog scaling factors. Signal is too bright. (max analog low: {al_max:.0f})")
            if (al_max < (al_upper_limit / 10000 * iterations)):
                raise Exception(f"Failed to adjust analog scaling factors. Signal is too dark. (max analog low: {al_max:.0f})")

            window_ms = np.round(al_upper_limit / al_max * window_ms, 3)

            report.write(f"window_ms ; {window_ms:.3f}\n")
            report.write('\n')
            
            cnt = {}    # The measured couting values.
            al = {}     # The measured analog low values.
            ah = {}     # The measured analog high values.
            
            output = f"signal ; "
            for ch in channel:
                cnt[ch] = np.zeros_like(led_current)
                al[ch] = np.zeros_like(led_current)
                ah[ch] = np.zeros_like(led_current)

                output += f"{ch + '_cnt':9} ; {ch + '_al':9} ; {ch + '_ah':9} ; "

            await send_to_gc(output, report=report)

            for i, current in enumerate(led_current):
                for ch in channel:
                    led_source, led_channel, led_type = led_dimmed[ch].split('_')
                    await set_led_current(current, led_source, led_channel, led_type)

                results = await pmt_multi_range_measurement(window_ms, iterations)
                if GlobalVar.get_stop_gc():
                    return f"pmt_adjust_analog stopped by user"

                output = f"{led_current[i]:6.0f} ; "
                for ch in channel:
                    cnt[ch][i] = results[f"{ch}_cnt_mean"]
                    al[ch][i] = results[f"{ch}_al_mean"]
                    ah[ch][i] = results[f"{ch}_ah_mean"]

                    output += f"{cnt[ch][i]:9.0f} ; {al[ch][i]:9.0f} ; {ah[ch][i]:9.0f} ; "
                
                await send_to_gc(output, report=report)
                
            report.write('\n')

            for ch in channel:
                led_source, led_channel, led_type = led_dimmed[ch].split('_')
                await set_led_current(0, led_source, led_channel, led_type)
        
            if isinstance(ppr_ns, float):
                ppr_ns = {'pmt1':ppr_ns, 'pmt2':ppr_ns, 'uslum':ppr_ns, 'htsal':ppr_ns}
            
            als = {}    # The adjusted analog low scaling.
            ahs = {}    # The adjsuted analog high scaling.

            for ch in channel:

                cnt[ch] = pmt_calculate_correction(cnt[ch], ppr_ns[ch], window_ms)

                limits = np.logical_and((al[ch] >= al_lower_limit), (al[ch] <= al_upper_limit))
                als[ch] = np.sum(cnt[ch][limits] * al[ch][limits]) / np.sum(al[ch][limits] ** 2)
                ahs[ch] = np.sum(al[ch][limits] * ah[ch][limits]) / np.sum(ah[ch][limits] ** 2)

                al[ch] = al[ch] * als[ch]
                ah[ch] = ah[ch] * ahs[ch] * als[ch]

            report.write(dict_to_str(als, key_suffix='_als', value_format='.6f', value_separator=' ; ', item_separator=' ; '))
            report.write('\n')
            report.write(dict_to_str(ahs, key_suffix='_ahs', value_format='.4f', value_separator=' ; ', item_separator=' ; '))
            report.write('\n')
            report.write('\n')

            await plot_analog_scan(channel, led_current, cnt, al, ah, file_name='analog_scan.png')

        return {'als':als, 'ahs':ahs}

    finally:
        await pmt_set_hv_enable(channel, 0)
        await base_tester_enable(False)


async def pmt_dark_signal(channel, dl, hv, window_ms=50, window_count=20, iterations=60, report_file=''):

    channel = parse_channel_parameter(channel, mask=('pmt1','pmt2','uslum','htsal'))

    report_file = str(report_file) if (report_file != '') else os.path.join(report_path, f"pmt_dark_signal.csv")

    await base_tester_enable(True)
    try:
        with open(report_file, 'a') as report:
            timestamp = datetime.now().strftime('%Y%m%d_%H%M%S')
            report.write(f"pmt_dark_signal(channel={channel}, dl={dl}, hv={hv}, window_ms={window_ms}, window_count={window_count}, iterations={iterations}) started at {timestamp}\n")
            report.write(f"temperature: {await pmt_get_temperature(channel)}\n")
            report.write('\n')

            dark_mean = {}
            dark_max = {}
            dark_std = {}
            dark_std_error = {}

            # Measure channels sequential as a quick fix for uslum and htsal being the same PMT with different settings.
            for ch in channel:
                await pmt_set_dl(ch, dl)
                await pmt_set_hv(ch, hv)
                await asyncio.sleep(pmt_set_hv_delay)
                await pmt_set_hv_enable(ch, 1)
                await asyncio.sleep(pmt_set_hv_enable_delay)

                results = await pmt_counting_measurement(window_ms, window_count, iterations)
                if GlobalVar.get_stop_gc():
                    return f"pmt_dark_signal stopped by user"

                dark_mean[ch] = results[f"{ch}_cps_mean"]
                dark_max[ch] = results[f"{ch}_cps_max"]
                dark_std[ch] = results[f"{ch}_cps_std"]
                dark_std_error[ch] = (dark_std[ch] - np.sqrt(dark_mean[ch])) / np.sqrt(dark_mean[ch])

            report.write(dict_to_str(dark_mean, key_suffix='_dark_mean', value_format='.1f', value_separator=' ; ', item_separator=' ; '))
            report.write('\n')
            report.write(dict_to_str(dark_max, key_suffix='_dark_max', value_format='.1f', value_separator=' ; ', item_separator=' ; '))
            report.write('\n')
            report.write(dict_to_str(dark_std, key_suffix='_dark_std', value_format='.3f', value_separator=' ; ', item_separator=' ; '))
            report.write('\n')
            report.write(dict_to_str(dark_std_error, key_suffix='_dark_std_error', value_format='.2%', value_separator=' ; ', item_separator=' ; '))
            report.write('\n')
            report.write('\n')

        return {'dark_mean':dark_mean, 'dark_max':dark_max, 'dark_std':dark_std, 'dark_std_error':dark_std_error}

    finally:
        await pmt_set_hv_enable(channel, 0)
        await base_tester_enable(False)


async def pdd_adjust_analog(channel, report_file=''):

    channel = parse_channel_parameter(channel, mask=('pmt1','pmt2','uslum','htsal'))
    
    iterations = 20

    led_current = np.linspace(1, 127, 127)
    
    al_lower_limit = round((65536 - 1310) * 0.2)    # The analog low value must be greater or equal to this limit. 20% of the analog low range with ~100 mV offset voltage.
    al_upper_limit = round((65536 - 1310) * 0.8)    # The analog low value must be less or equal to this limit. 80% of the analog low range with ~100 mV offset voltage.
    
    report_file = str(report_file) if (report_file != '') else os.path.join(report_path, f"pdd_adjust_analog.csv")

    await base_tester_enable(True)
    try:
        with open(report_file, 'a') as report:
            timestamp = datetime.now().strftime('%Y%m%d_%H%M%S')
            report.write(f"pdd_adjust_analog(channel={channel}) started at {timestamp}\n")
            report.write('\n')
            report.write(f"parameter: iterations={iterations}, al_lower_limit={al_lower_limit}, al_upper_limit={al_upper_limit}\n")
            report.write('\n')
            
            # Determine the required measurement window time in ms.
            window_ms = 10.0
            for ch in channel:
                led_source, led_channel, led_type = led_dimmed[ch].split('_')
                await set_led_current(np.max(led_current), led_source, led_channel, led_type)

            results = await pdd_multi_range_measurement(window_ms, iterations)

            al_max = 0.0
            for ch in channel:
                al_max = max(al_max, results[f"{ch}_pdd_al_mean"])
            if (al_max > al_upper_limit):
                window_ms = 1.0
                results = await pdd_multi_range_measurement(window_ms, iterations)
            
            al_max = 0.0
            for ch in channel:
                al_max = max(al_max, results[f"{ch}_pdd_al_mean"])
            if (al_max > al_upper_limit):
                raise Exception(f"Failed to adjust analog scaling factors. Signal is too bright. (max analog low: {al_max:.0f})")
            if (al_max < (al_upper_limit / 10000 * iterations)):
                raise Exception(f"Failed to adjust analog scaling factors. Signal is too dark. (max analog low: {al_max:.0f})")

            window_ms = np.round(al_upper_limit / al_max * window_ms, 3)

            report.write(f"window_ms ; {window_ms:.3f}\n")
            report.write('\n')
            
            al = {}     # The measured analog low values.
            ah = {}     # The measured analog high values.
            
            output = f"signal ; "
            for ch in channel:
                al[ch] = np.zeros_like(led_current)
                ah[ch] = np.zeros_like(led_current)

                output += f"{ch + '_pdd_al':12} ; {ch + '_pdd_ah':12} ; "

            await send_to_gc(output, log=True, report=report)

            for i, current in enumerate(led_current):
                for ch in channel:
                    led_source, led_channel, led_type = led_dimmed[ch].split('_')
                    await set_led_current(current, led_source, led_channel, led_type)

                results = await pdd_multi_range_measurement(window_ms, iterations)
                if GlobalVar.get_stop_gc():
                    return f"pdd_adjust_analog stopped by user"

                output = f"{led_current[i]:6.0f} ; "
                for ch in channel:
                    al[ch][i] = results[f"{ch}_pdd_al_mean"]
                    ah[ch][i] = results[f"{ch}_pdd_ah_mean"]

                    output += f"{al[ch][i]:12.0f} ; {ah[ch][i]:12.0f} ; "
                
                await send_to_gc(output, log=True, report=report)
                
            report.write('\n')

            for ch in channel:
                led_source, led_channel, led_type = led_dimmed[ch].split('_')
                await set_led_current(0, led_source, led_channel, led_type)
            
            ahs = {}    # The adjsuted analog high scaling.

            for ch in channel:
                limits = np.logical_and((al[ch] >= al_lower_limit), (al[ch] <= al_upper_limit))
                ahs[ch] = np.sum(al[ch][limits] * ah[ch][limits]) / np.sum(ah[ch][limits] ** 2)

                ah[ch] = ah[ch] * ahs[ch]

            report.write(dict_to_str(ahs, key_suffix='_pdd_scaling', value_separator=' ; ', item_separator=' ; '))
            report.write('\n')
            report.write('\n')

            await plot_analog_scan(channel, led_current, {ch:None for ch in channel}, al, ah, file_name='analog_scan.png')

        return ahs
    
    finally:
        await base_tester_enable(False)


async def pdd_adjust_led_scaling(channel, dl, hv, ppr_ns, report_file=''):

    channel = parse_channel_parameter(channel, mask=('pmt1','pmt2','uslum','htsal'))

    window_count = 10
    
    led_current_dimmed = [18,24,33,45,63,88,124,127]
    led_current_bright = [1,2,3,4,5,7,10,13]

    scan_size = len(led_current_dimmed) + len(led_current_bright)

    pdd_upper_limit = round((65536 - 1310) * 101.0 * 0.8)   # The analog value must be less or equal to this limit. 80% of the analog high range with ~100 mV offset voltage.

    pmt_signal = {}     # The measured signal values of the pmt in counts per second.
    pdd_signal = {}     # The measured signal values of the photodiode.
    
    report_file = str(report_file) if (report_file != '') else os.path.join(report_path, f"pdd_adjust_led_scaling.csv")

    await base_tester_enable(True)
    try:
        with open(report_file, 'a') as report:
            timestamp = datetime.now().strftime('%Y%m%d_%H%M%S')
            report.write(f"pdd_adjust_led_scaling(channel={channel}, dl={dl}, hv={hv}, ppr_ns={ppr_ns}) started at {timestamp}\n")
            report.write(f"temperature: {await pmt_get_temperature(channel)}\n")
            report.write('\n')
            report.write(f"parameter: window_count={window_count}, led_current_dimmed={led_current_dimmed}, led_current_bright={led_current_bright}, pdd_upper_limit={pdd_upper_limit}\n")
            report.write('\n')

            for led in np.unique(list(led_dimmed.values()) + list(led_bright.values())):
                led_source, led_channel, led_type = led.split('_')
                await set_led_current(0, led_source, led_channel, led_type)

            await pmt_set_dl(channel, dl)
            await pmt_set_hv(channel, hv)
            await asyncio.sleep(pmt_set_hv_delay)
            await pmt_set_hv_enable(channel, 1)
            await asyncio.sleep(pmt_set_hv_enable_delay)

            # Determine the required measurement window time in ms.
            window_ms = 10.0
            for ch in channel:
                led_source, led_channel, led_type = led_bright[ch].split('_')
                await set_led_current(np.max(led_current_bright), led_source, led_channel, led_type)

            results = await pdd_analog_measurement(window_ms, iterations=window_count)

            pdd_max = 0.0
            for ch in channel:
                pdd_max = max(pdd_max, results[f"{ch}_pdd_mean"])
            if (pdd_max > pdd_upper_limit):
                window_ms = 1.0
                results = await pmt_counting_measurement(window_ms, window_count, iterations=1)

            for ch in channel:
                led_source, led_channel, led_type = led_bright[ch].split('_')
                await set_led_current(0, led_source, led_channel, led_type)

            pdd_max = 0.0
            for ch in channel:
                pdd_max = max(pdd_max, results[f"{ch}_pdd_mean"])
            if (pdd_max > pdd_upper_limit):
                raise Exception(f"Failed to adjust LED scaling factors. Signal is too bright. (max analog: {pdd_max:.0f})")

            window_ms = np.round(pdd_upper_limit / pdd_max * window_ms, 3)

            report.write(f"window_ms ; {window_ms:.3f}\n")
            report.write('\n')

            output = ''
            for ch in channel:
                pmt_signal[ch] = np.zeros(scan_size)
                pdd_signal[ch] = np.zeros(scan_size)

                output += f"{ch + '_led':14} ; current ; {ch + '_pmt':11} ; {ch + '_pdd':11} ; "

            await send_to_gc(output, log=True, report=report)

            index = 0

            for current in led_current_dimmed:
                for ch in channel:
                    led_source, led_channel, led_type = led_dimmed[ch].split('_')
                    await set_led_current(current, led_source, led_channel, led_type)

                results = await pmt_counting_measurement(window_ms, window_count, iterations=1)
                if GlobalVar.get_stop_gc():
                    return f"pdd_adjust_led_scaling stopped by user"

                output = ''
                for ch in channel:
                    pmt_signal[ch][index] = results[f"{ch}_cps_mean"]
                    pdd_signal[ch][index] = results[f"{ch}_pdd_mean"]

                    output += f"{led_dimmed[ch]:14} ; {current:7.0f} ; {pmt_signal[ch][index]:11.0f} ; {pdd_signal[ch][index]:11.0f} ; "

                await send_to_gc(output, log=True, report=report)

                index += 1
            
            for ch in channel:
                led_source, led_channel, led_type = led_dimmed[ch].split('_')
                await set_led_current(0, led_source, led_channel, led_type)

            for current in led_current_bright:
                for ch in channel:
                    led_source, led_channel, led_type = led_bright[ch].split('_')
                    await set_led_current(current, led_source, led_channel, led_type)

                results = await pmt_counting_measurement(window_ms, window_count, iterations=1)
                if GlobalVar.get_stop_gc():
                    return f"pdd_adjust_led_scaling stopped by user"

                output = ''
                for ch in channel:
                    pmt_signal[ch][index] = results[f"{ch}_cps_mean"]
                    pdd_signal[ch][index] = results[f"{ch}_pdd_mean"]

                    output += f"{led_bright[ch]:14} ; {current:7.0f} ; {pmt_signal[ch][index]:11.0f} ; {pdd_signal[ch][index]:11.0f} ; "

                await send_to_gc(output, log=True, report=report)

                index += 1

            report.write('\n')
            
            for ch in channel:
                led_source, led_channel, led_type = led_bright[ch].split('_')
                await set_led_current(0, led_source, led_channel, led_type)

            led_scaling = {}
            pdd_scaled = {}
        
            if isinstance(ppr_ns, float):
                ppr_ns = {'pmt1':ppr_ns, 'pmt2':ppr_ns, 'uslum':ppr_ns, 'htsal':ppr_ns}

            for ch in channel:

                pmt_signal[ch] = pmt_calculate_correction(pmt_signal[ch], ppr_ns[ch])
                
                slope_dimmed = np.mean(pdd_signal[ch][:len(led_current_dimmed)] / pmt_signal[ch][:len(led_current_dimmed)])
                slope_bright = np.mean(pdd_signal[ch][len(led_current_dimmed):] / pmt_signal[ch][len(led_current_dimmed):])

                led_scaling[ch] = slope_bright / slope_dimmed

                report.write(f"{ch}_led_scaling ; {led_scaling[ch]:.6f} ; ")

                pdd_scaled[ch] = pdd_signal[ch].copy()
                pdd_scaled[ch][:len(led_current_dimmed)] *= led_scaling[ch]

            report.write(dict_to_str(led_scaling, key_suffix='_led_scaling', value_format='.6f', value_separator=' ; ', item_separator=' ; '))
            report.write('\n')
            report.write('\n')

            await plot_pdd_scan(channel, pmt_signal, pdd_signal, pdd_scaled, file_name=f"pdd_scan.png")

        return led_scaling
    
    finally:
        await pmt_set_hv_enable(channel, 0)
        await base_tester_enable(False)


async def plot_dl_scan(channel, dl_range, cps, dt, dl, file_name='graph.png'):

    plt.clf()

    for i, ch in enumerate(channel):
        plt.subplot(len(channel), 2, (i * 2 + 1))
        if i == 0:
            plt.title('Counts Per Second')
        if i == (len(channel) - 1):
            plt.xlabel('discriminator level')
        plt.ylabel(ch.upper())
        plt.yscale('symlog', linthresh=1)
        plt.plot(dl_range, cps[ch])
        plt.axvline(dl[ch], color='r')

        plt.subplot(len(channel), 2, (i * 2 + 2))
        if i == 0:
            plt.title('Dead Time')
        if i == (len(channel) - 1):
            plt.xlabel('discriminator level')
        plt.plot(dl_range, dt[ch])
        plt.axvline(dl[ch], color='r')

    plt.savefig(os.path.join(images_path, file_name))
    await send_gc_event('RefreshGraph', file_name=os.path.join('pmt_adjust_images', file_name))


async def plot_signal_scan(channel, signal, cps, cps_ref, hv, file_name='graph.png'):
    
    if isinstance(hv, float):
        hv = {'pmt1':hv, 'pmt2':hv, 'uslum':hv, 'htsal':hv}

    plt.clf()

    for i, ch in enumerate(channel):
        plt.subplot(len(channel), 1, (i + 1))
        if i == 0:
            plt.title(f"Signal Scan HV={hv[ch]:.3f}")
        if i == (len(channel) - 1):
            plt.xlabel('signal')
        plt.ylabel(ch.upper())
        plt.xscale('log')
        plt.yscale('log')
        plt.plot(signal[ch], cps_ref[ch], label='reference', color='r')
        plt.plot(signal[ch], cps[ch], label='measured', color='b')
        plt.legend(loc='upper left')

    plt.savefig(os.path.join(images_path, file_name))
    await send_gc_event('RefreshGraph', file_name=os.path.join('pmt_adjust_images', file_name))


async def plot_hv_scan(channel, hv_range, scan, hv, title='HV Scan', file_name='graph.png'):

    plt.clf()

    for i, ch in enumerate(channel):
        plt.subplot(len(channel), 1, (i + 1))
        if i == 0:
            plt.title(title)
        if i == (len(channel) - 1):
            plt.xlabel('HV')
        plt.ylabel(ch.upper())
        plt.plot(hv_range, scan[ch])
        plt.axvline(hv[ch], color='r')

    plt.savefig(os.path.join(images_path, file_name))
    await send_gc_event('RefreshGraph', file_name=os.path.join('pmt_adjust_images', file_name))


async def plot_analog_scan(channel, current, cnt, al, ah, file_name='graph.png'):

    plt.clf()

    for i, ch in enumerate(channel):
        plt.subplot(len(channel), 1, (i + 1))
        if i == 0:
            plt.title(f"Analog Adjustment")
        plt.ylabel(ch.upper())
        if ah[ch] is not None:
            plt.plot(current, ah[ch], label='analog_high', color='g')
        if al[ch] is not None:
            plt.plot(current, al[ch], label='analog_low', color='r')
        if cnt[ch] is not None:
            plt.plot(current, cnt[ch], label='counting', color='b')
        plt.legend(loc='upper left')

    plt.savefig(os.path.join(images_path, file_name))
    await send_gc_event('RefreshGraph', file_name=os.path.join('pmt_adjust_images', file_name))


async def plot_pdd_scan(channel, pmt_signal, pdd_measured, pdd_scaled, file_name='graph.png'):

    plt.clf()

    for i, ch in enumerate(channel):
        plt.subplot(len(channel), 1, (i + 1))
        if i == 0:
            plt.title(f"Photodiode Signal Scan")
        if i == (len(channel) - 1):
            plt.xlabel('pmt_signal')
        plt.ylabel(f"PDD of {ch.upper()}")
        plt.xscale('log')
        plt.yscale('log')
        plt.plot(pmt_signal[ch], pdd_measured[ch], label='pdd_measured', color='r')
        plt.plot(pmt_signal[ch], pdd_scaled[ch], label='pdd_scaled', color='b')
        plt.legend(loc='upper left')

    plt.savefig(os.path.join(images_path, file_name))
    await send_gc_event('RefreshGraph', file_name=os.path.join('pmt_adjust_images', file_name))


def dict_to_str(dict, key_format='', key_suffix='', value_format='', value_suffix='', value_separator=' = ', item_separator=', '):
    result = ''
    for key, value in dict.items():
        result += f"{key:{key_format}}{key_suffix}{value_separator}{value:{value_format}}{value_suffix}{item_separator}"

    return result[:-len(item_separator)]


def pmt_calculate_ppr_ns(signal, counts, window_ms=1000, optimization='r_squared_min', optimization_arg=None):
    ppr_ns_max = 25.0
    ppr_ns = 0.0

    linearity, rel_error = pmt_calculate_linearity(signal, counts, ppr_ns, window_ms)

    if optimization == 'none':
        return (ppr_ns, np.min(linearity), np.max(rel_error))

    precision = 2
    for step_ns in [1.0 / 10**digit for digit in range(precision + 1)]:

        if ppr_ns >= (step_ns * 10.0):
            # Take one step back for cases like this: Optimal ppr is 11.5 and r²(11.0) < r²(12.0) > r²(13.0).
            ppr_ns = round((ppr_ns - step_ns * 10.0), precision)
            linearity, rel_error = pmt_calculate_linearity(signal, counts, ppr_ns, window_ms)

        stop = False
        while not stop:
            temp_ppr_ns = round((ppr_ns + step_ns), precision)
            temp_linearity, temp_rel_error = pmt_calculate_linearity(signal, counts, temp_ppr_ns, window_ms)

            if optimization == 'r_squared':
                index = int(optimization_arg) if (optimization_arg is not None) else (len(signal) - 1)
                if (index < 2) or (index >= len(signal)):
                    raise ValueError(f"Invalid optimization signal index.")
                stop = temp_linearity[index] < linearity[index]
            elif optimization == 'r_squared_min':
                stop = np.min(temp_linearity) < np.min(linearity)
            elif optimization == 'rel_error_max':
                stop = np.max(temp_rel_error) > np.max(rel_error)
            else:
                raise ValueError(f"Invalid optimization option.")

            if not stop:
                ppr_ns = temp_ppr_ns
                linearity = temp_linearity
                rel_error = temp_rel_error

            if ppr_ns > ppr_ns_max:
                # Failed to calculate pulse pair resolution
                ppr_ns = 0.0
                linearity, rel_error = pmt_calculate_linearity(signal, counts, ppr_ns, window_ms)
                return (ppr_ns, np.min(linearity), np.max(rel_error))

    return (ppr_ns, np.min(linearity), np.max(rel_error))


def pmt_calculate_linearity(signal, counts, ppr_ns, window_ms=1000):

    counts = pmt_calculate_correction(counts, ppr_ns, window_ms)

    linearity = pmt_calculate_r_squared(signal, counts)
    rel_error = pmt_calculate_rel_error(signal, counts)

    return (linearity, rel_error)


def pmt_calculate_r_squared(signal, counts):

    r_squared = np.ones_like(signal)

    for i in range(2, len(signal)):
        correlation_coef = np.corrcoef(signal[:(i + 1)], counts[:(i + 1)])
        r_value = correlation_coef[0,1]
        r_squared[i] = r_value**2

    return r_squared


def pmt_calculate_rel_error(signal, counts):

    ref_size = round(len(signal) / 2)
    
    counts_ref = np.mean(counts[:ref_size] / signal[:ref_size]) * signal
    rel_error = np.abs((counts - counts_ref) / counts_ref)

    return rel_error


def pmt_calculate_offset(signal, counts):

    trendline = np.polyfit(signal, counts, deg=1)
    counts_ref = np.polyval(trendline, signal)

    offset = np.abs(counts - counts_ref) / signal

    return offset


def pmt_calculate_correction(counts, ppr_ns, window_ms=1000):

    counts = np.array(counts)
    counts = counts / (1 - counts * ppr_ns * 1e-6 / window_ms)

    return counts


def binomial_filter(data, iterations=1):
    data_size = len(data)
    for _ in range(iterations):
        smoothed_data = np.zeros(data_size)
        for i in range(data_size):
            if i == 0:
                smoothed_data[i] = ((data[i] * 3 + data[i + 1]) / 4)
            elif i == (data_size - 1):
                smoothed_data[i] = ((data[i] * 3 + data[i - 1]) / 4)
            else:
                smoothed_data[i] = ((data[i] * 2 + data[i + 1] + data[i - 1]) / 4)
        
        data = smoothed_data

    return data


def parse_channel_parameter(channel, mask=('pmt1','pmt2','uslum','htsal')):
    return [ch for ch in mask if ch in channel]


base_tester_is_enabled = 0

async def base_tester_enable(enable=True, nested=True):
    global base_tester_is_enabled

    if nested and enable:
        base_tester_is_enabled += 1
    elif nested and not enable:
        base_tester_is_enabled -= 1
    else:
        base_tester_is_enabled = int(enable == True)

    if base_tester_is_enabled == 1:
        try:
            await fmb_unit.StartFirmware()
            await fmb_unit.SetDigitalOutput(FMBDigitalOutput.PSUON, 0)
            await asyncio.sleep(0.1)
            await asyncio.gather(
                eef_unit.StartFirmware(),
                mc6_unit.StartFirmware(),
            )
            await fmb_endpoint.SetParameter(NodeParameter.LEDPatternEnable, 0)
            await eef_endpoint.SetParameter(NodeParameter.LEDPatternEnable, 0)
            await mc6_endpoint.SetParameter(NodeParameter.LEDPatternEnable, 0)
            await base_fan.InitializeDevice()
            await base_fan.enable()
        except Exception as ex:
            base_tester_is_enabled = 0
            await fmb_unit.endpoint.Reset()
            raise
    elif base_tester_is_enabled < 1:
        base_tester_is_enabled = 0
        await fmb_unit.endpoint.Reset()

    return base_tester_is_enabled


async def wait_for_interlock(state, timeout=30):
    remaining = timeout
    while not GlobalVar.get_stop_gc() and (await get_interlock_state() != state):
        if remaining > 0:
            await asyncio.sleep(0.1)
            remaining -= 0.1
        else:
            raise Exception(f"Waiting for interlock state '{state}' within {timeout}s failed.")


async def get_interlock_state():
    for retry in range(3):
        interlock_status = (await meas_endpoint.GetParameter(MeasurementParameter.InterlockStatus))[0]
        if interlock_status == 0b0000:
            return 'closed'
        if interlock_status == 0b0101:
            return 'open'
        await asyncio.sleep(0.01)
        
    raise Exception(f"Invalid interlock status = 0b{interlock_status:04b}")


async def pmt_get_temperature(channel):
    temperature = ''
    if 'pmt1' in channel:
        temperature += f"pmt1={await pmt1_cooling.get_feedback_value():.2f}°C, "
    if 'pmt2' in channel:
        temperature += f"pmt2={await pmt2_cooling.get_feedback_value():.2f}°C, "
    if 'htsal' in channel:
        temperature += f"htsal={await hts_alpha_cooling.get_feedback_value():.2f}°C, "
    
    return temperature[:-2] if len(temperature) > 2 else temperature


async def pmt_set_dl(channel, dl):
    if isinstance(dl, float):
        dl = {'pmt1':dl, 'pmt2':dl, 'uslum':dl, 'htsal':dl}
    if 'pmt1' in channel:
        await meas_endpoint.SetParameter(MeasurementParameter.PMT1DiscriminatorLevel, dl['pmt1'])
    if 'pmt2' in channel:
        await meas_endpoint.SetParameter(MeasurementParameter.PMT2DiscriminatorLevel, dl['pmt2'])
    if 'uslum' in channel:
        await meas_endpoint.SetParameter(MeasurementParameter.PMTUSLUMDiscriminatorLevel, dl['uslum'])
    if 'htsal' in channel:
        await meas_endpoint.SetParameter(MeasurementParameter.PMTUSLUMDiscriminatorLevel, dl['htsal'])


async def pmt_set_hv(channel, hv):
    if isinstance(hv, float):
        hv = {'pmt1':hv, 'pmt2':hv, 'uslum':hv, 'htsal':hv}
    if 'pmt1' in channel:
        await meas_endpoint.SetParameter(MeasurementParameter.PMT1HighVoltageSetting, hv['pmt1'])
    if 'pmt2' in channel:
        await meas_endpoint.SetParameter(MeasurementParameter.PMT2HighVoltageSetting, hv['pmt2'])
    if 'uslum' in channel:
        await meas_endpoint.SetParameter(MeasurementParameter.PMTUSLUMHighVoltageSetting, hv['uslum'])
    if 'htsal' in channel:
        await meas_endpoint.SetParameter(MeasurementParameter.PMTUSLUMHighVoltageSetting, hv['htsal'])


async def pmt_set_hv_enable(channel, enable):
    if 'pmt1' in channel:
        await meas_endpoint.SetParameter(MeasurementParameter.PMT1HighVoltageEnable, enable)
    if 'pmt2' in channel:
        await meas_endpoint.SetParameter(MeasurementParameter.PMT2HighVoltageEnable, enable)
    if 'uslum' in channel:
        await meas_endpoint.SetParameter(MeasurementParameter.PMTUSLUMHighVoltageEnable, enable)
    if 'htsal' in channel:
        await meas_endpoint.SetParameter(MeasurementParameter.PMTUSLUMHighVoltageEnable, enable)


async def pmt_counting_measurement(window_ms=50.0, window_count=20, iterations=1):
    
    op_id = 'pmt_counting_measurement'
    meas_unit.ClearOperations()
    await load_pmt_counting_measurement(op_id, window_ms, window_count)

    pmt1_pdd_scaling = config.get('pmt1_pdd_scaling', 101.0)
    pmt2_pdd_scaling = config.get('pmt2_pdd_scaling', 101.0)
    uslum_pdd_scaling = config.get('uslum_pdd_scaling', 101.0)
    htsal_pdd_scaling = config.get('htsal_pdd_scaling', 101.0)
    
    pmt1_cps = np.zeros(iterations)
    pmt1_dt = np.zeros(iterations)
    pmt1_pdd = np.zeros(iterations)
    pmt2_cps = np.zeros(iterations)
    pmt2_dt = np.zeros(iterations)
    pmt2_pdd = np.zeros(iterations)
    uslum_cps = np.zeros(iterations)
    uslum_dt = np.zeros(iterations)
    uslum_pdd = np.zeros(iterations)
    htsal_cps = np.zeros(iterations)
    htsal_dt = np.zeros(iterations)
    htsal_pdd = np.zeros(iterations)

    for i in range(iterations):
        if GlobalVar.get_stop_gc():
            return f"pmt_counting_measurement stopped by user"
        
        await meas_unit.ExecuteMeasurement(op_id)
        results = await meas_unit.ReadMeasurementValues(op_id)

        pmt1_cps[i] = (results[0]  + (results[1]  << 32)) / window_count / window_ms * 1000.0
        pmt1_dt[i]  = (results[2]  + (results[3]  << 32)) / window_count / window_ms * 1000.0
        pmt2_cps[i] = (results[6]  + (results[7]  << 32)) / window_count / window_ms * 1000.0
        pmt2_dt[i]  = (results[8]  + (results[9]  << 32)) / window_count / window_ms * 1000.0
        uslum_cps[i] = (results[12] + (results[13] << 32)) / window_count / window_ms * 1000.0
        uslum_dt[i]  = (results[14] + (results[15] << 32)) / window_count / window_ms * 1000.0
        htsal_cps[i] = (results[12] + (results[13] << 32)) / window_count / window_ms * 1000.0
        htsal_dt[i]  = (results[14] + (results[15] << 32)) / window_count / window_ms * 1000.0
        pmt1_pdd[i] = (results[16] + results[17] * pmt1_pdd_scaling) / window_count / window_ms * 1000.0
        pmt2_pdd[i] = (results[18] + results[19] * pmt2_pdd_scaling) / window_count / window_ms * 1000.0
        uslum_pdd[i] = (results[20] + results[21] * uslum_pdd_scaling) / window_count / window_ms * 1000.0
        htsal_pdd[i] = (results[20] + results[21] * htsal_pdd_scaling) / window_count / window_ms * 1000.0

    results = {}

    results['pmt1_cps_mean'] = np.mean(pmt1_cps)
    results['pmt1_dt_mean'] = np.mean(pmt1_dt)
    results['pmt1_pdd_mean'] = np.mean(pmt1_pdd)
    results['pmt2_cps_mean'] = np.mean(pmt2_cps)
    results['pmt2_dt_mean'] = np.mean(pmt2_dt)
    results['pmt2_pdd_mean'] = np.mean(pmt2_pdd)
    results['uslum_cps_mean'] = np.mean(uslum_cps)
    results['uslum_dt_mean'] = np.mean(uslum_dt)
    results['uslum_pdd_mean'] = np.mean(uslum_pdd)
    results['htsal_cps_mean'] = np.mean(htsal_cps)
    results['htsal_dt_mean'] = np.mean(htsal_dt)
    results['htsal_pdd_mean'] = np.mean(htsal_pdd)
    
    results['pmt1_cps_std'] = np.std(pmt1_cps)
    results['pmt1_dt_std'] = np.std(pmt1_dt)
    results['pmt1_pdd_std'] = np.std(pmt1_pdd)
    results['pmt2_cps_std'] = np.std(pmt2_cps)
    results['pmt2_dt_std'] = np.std(pmt2_dt)
    results['pmt2_pdd_std'] = np.std(pmt2_pdd)
    results['uslum_cps_std'] = np.std(uslum_cps)
    results['uslum_dt_std'] = np.std(uslum_dt)
    results['uslum_pdd_std'] = np.std(uslum_pdd)
    results['htsal_cps_std'] = np.std(htsal_cps)
    results['htsal_dt_std'] = np.std(htsal_dt)
    results['htsal_pdd_std'] = np.std(htsal_pdd)
    
    results['pmt1_cps_max'] = np.max(pmt1_cps)
    results['pmt1_dt_max'] = np.max(pmt1_dt)
    results['pmt1_pdd_max'] = np.max(pmt1_pdd)
    results['pmt2_cps_max'] = np.max(pmt2_cps)
    results['pmt2_dt_max'] = np.max(pmt2_dt)
    results['pmt2_pdd_max'] = np.max(pmt2_pdd)
    results['uslum_cps_max'] = np.max(uslum_cps)
    results['uslum_dt_max'] = np.max(uslum_dt)
    results['uslum_pdd_max'] = np.max(uslum_pdd)
    results['htsal_cps_max'] = np.max(htsal_cps)
    results['htsal_dt_max'] = np.max(htsal_dt)
    results['htsal_pdd_max'] = np.max(htsal_pdd)

    return results


async def load_pmt_counting_measurement(op_id, window_ms, window_count):
    if (window_ms < 0.02):
        raise ValueError(f"window_ms must be greater or equal to 0.02 ms")

    fixed_range_us = 20
    auto_range_us = round(window_ms * 1000) - fixed_range_us
    auto_range_us_coarse, auto_range_us_fine = divmod(auto_range_us, 65536)

    hv_gate_delay = 1000000     #  10 ms
    full_reset_delay = 40000    # 400 us
    us_tick_delay = 100         #   1 us
    conversion_delay = 1200     #  12 us
    switch_delay = 25           # 250 ns

    seq_gen = meas_seq_generator()

    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 0)  # pmt1_cnt_lsb
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 1)  # pmt1_cnt_msb
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 2)  # pmt1_dt_lsb
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 3)  # pmt1_dt_msb
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 4)  # pmt1_al
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 5)  # pmt1_ah
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 6)  # pmt2_cnt_lsb
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 7)  # pmt2_cnt_msb
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 8)  # pmt2_dt_lsb
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 9)  # pmt2_dt_msb
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=10)  # pmt2_al
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=11)  # pmt2_ah
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=12)  # pmt3_cnt_lsb
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=13)  # pmt3_cnt_msb
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=14)  # pmt3_dt_lsb
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=15)  # pmt3_dt_msb
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=16)  # ref_al
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=17)  # ref_ah
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=18)  # abs_al
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=19)  # abs_ah
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=20)  # aux_al
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=21)  # aux_ah

    seq_gen.TimerWaitAndRestart(hv_gate_delay)
    seq_gen.ResetSignals(OutputSignal.InputGatePMT1 | OutputSignal.InputGatePMT2)
    seq_gen.SetSignals(OutputSignal.HVGatePMT1 | OutputSignal.HVGatePMT2)

    seq_gen.Loop(window_count)
    seq_gen.SetAnalogControl(ref=AnalogControlMode.full_offset_reset, abs=AnalogControlMode.full_offset_reset, aux=AnalogControlMode.full_offset_reset)

    seq_gen.TimerWaitAndRestart(full_reset_delay)
    seq_gen.SetIntegratorMode(ref=IntegratorMode.full_reset, abs=IntegratorMode.full_reset, aux=IntegratorMode.full_reset)

    seq_gen.TimerWaitAndRestart(conversion_delay)
    seq_gen.SetTriggerOutput(TriggerSignal.SampleRef | TriggerSignal.SampleAbs | TriggerSignal.SampleAux)

    seq_gen.TimerWaitAndRestart(switch_delay)
    seq_gen.SetIntegratorMode(ref=IntegratorMode.low_range_reset, abs=IntegratorMode.low_range_reset, aux=IntegratorMode.low_range_reset)
    seq_gen.SetAnalogControl(ref=AnalogControlMode.read_offset, abs=AnalogControlMode.read_offset, aux=AnalogControlMode.read_offset)

    seq_gen.TimerWaitAndRestart(us_tick_delay)
    seq_gen.PulseCounterControl(MeasurementChannel.PMT1, cumulative=False, resetCounter=True, resetPresetCounter=True, correctionOn=False)
    seq_gen.PulseCounterControl(MeasurementChannel.PMT2, cumulative=False, resetCounter=True, resetPresetCounter=True, correctionOn=False)
    seq_gen.PulseCounterControl(MeasurementChannel.US_LUM, cumulative=False, resetCounter=True, resetPresetCounter=True, correctionOn=False)
    seq_gen.SetTriggerOutput(TriggerSignal.SampleRef | TriggerSignal.SampleAbs | TriggerSignal.SampleAux)
    seq_gen.SetIntegratorMode(ref=IntegratorMode.integrate_autorange, abs=IntegratorMode.integrate_autorange, aux=IntegratorMode.integrate_autorange)
    if auto_range_us_coarse > 0:
        seq_gen.Loop(auto_range_us_coarse)
        seq_gen.Loop(65536)
        seq_gen.TimerWaitAndRestart(us_tick_delay)
        seq_gen.PulseCounterControl(MeasurementChannel.PMT1, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=True)
        seq_gen.PulseCounterControl(MeasurementChannel.PMT2, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=True)
        seq_gen.PulseCounterControl(MeasurementChannel.US_LUM, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=True)
        seq_gen.LoopEnd()
        seq_gen.LoopEnd()
    if auto_range_us_fine > 0:
        seq_gen.Loop(auto_range_us_fine)
        seq_gen.TimerWaitAndRestart(us_tick_delay)
        seq_gen.PulseCounterControl(MeasurementChannel.PMT1, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=True)
        seq_gen.PulseCounterControl(MeasurementChannel.PMT2, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=True)
        seq_gen.PulseCounterControl(MeasurementChannel.US_LUM, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=True)
        seq_gen.LoopEnd()
        
    seq_gen.SetIntegratorMode(ref=IntegratorMode.integrate_with_fixed_range, abs=IntegratorMode.integrate_with_fixed_range, aux=IntegratorMode.integrate_with_fixed_range)
    seq_gen.SetAnalogControl(ref=AnalogControlMode.read_offset, abs=AnalogControlMode.read_offset, aux=AnalogControlMode.read_offset)
    
    seq_gen.Loop(fixed_range_us)
    seq_gen.TimerWaitAndRestart(us_tick_delay)
    seq_gen.PulseCounterControl(MeasurementChannel.PMT1, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=True)
    seq_gen.PulseCounterControl(MeasurementChannel.PMT2, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=True)
    seq_gen.PulseCounterControl(MeasurementChannel.US_LUM, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=True)
    seq_gen.LoopEnd()
    
    seq_gen.SetTriggerOutput(TriggerSignal.SampleRef | TriggerSignal.SampleAbs | TriggerSignal.SampleAux)

    seq_gen.GetPulseCounterResult(MeasurementChannel.PMT1, deadTime=False, relative=False, resetCounter=False, cumulative=True, dword=True, addrPos=0, resultPos=0)
    seq_gen.GetPulseCounterResult(MeasurementChannel.PMT1, deadTime=True, relative=False, resetCounter=True, cumulative=True, dword=True, addrPos=0, resultPos=2)
    seq_gen.GetPulseCounterResult(MeasurementChannel.PMT2, deadTime=False, relative=False, resetCounter=False, cumulative=True, dword=True, addrPos=0, resultPos=6)
    seq_gen.GetPulseCounterResult(MeasurementChannel.PMT2, deadTime=True, relative=False, resetCounter=True, cumulative=True, dword=True, addrPos=0, resultPos=8)
    seq_gen.GetPulseCounterResult(MeasurementChannel.US_LUM, deadTime=False, relative=False, resetCounter=False, cumulative=True, dword=True, addrPos=0, resultPos=12)
    seq_gen.GetPulseCounterResult(MeasurementChannel.US_LUM, deadTime=True, relative=False, resetCounter=True, cumulative=True, dword=True, addrPos=0, resultPos=14)
    
    seq_gen.GetAnalogResult(MeasurementChannel.REF, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=16)
    seq_gen.GetAnalogResult(MeasurementChannel.REF, isRelativeAddr=False, ignoreRange=False, isHiRange=True, addResult=True, dword=False, addrPos=0, resultPos=17)
    seq_gen.GetAnalogResult(MeasurementChannel.ABS, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=18)
    seq_gen.GetAnalogResult(MeasurementChannel.ABS, isRelativeAddr=False, ignoreRange=False, isHiRange=True, addResult=True, dword=False, addrPos=0, resultPos=19)
    seq_gen.GetAnalogResult(MeasurementChannel.AUX, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=20)
    seq_gen.GetAnalogResult(MeasurementChannel.AUX, isRelativeAddr=False, ignoreRange=False, isHiRange=True, addResult=True, dword=False, addrPos=0, resultPos=21)

    seq_gen.LoopEnd()

    seq_gen.ResetSignals(OutputSignal.HVGatePMT1 | OutputSignal.HVGatePMT2)
    seq_gen.SetAnalogControl(ref=AnalogControlMode.full_offset_reset, abs=AnalogControlMode.full_offset_reset, aux=AnalogControlMode.full_offset_reset)
    seq_gen.SetIntegratorMode(ref=IntegratorMode.full_reset, abs=IntegratorMode.full_reset, aux=IntegratorMode.full_reset)
    seq_gen.Stop(0)

    meas_unit.resultAddresses[op_id] = range(0, 22)
    await meas_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)


async def pmt_analog_measurement(window_ms=1.0, iterations=100):
    
    op_id = 'pmt_analog_measurement'
    meas_unit.ClearOperations()
    await load_pmt_analog_measurement(op_id, window_ms, iterations)
        
    await meas_unit.ExecuteMeasurement(op_id)
    results = await meas_unit.ReadMeasurementValues(op_id)

    pmt1_al  = results[4]
    pmt1_ah  = results[5]
    pmt2_al  = results[10]
    pmt2_ah  = results[11]

    results = {}

    results['pmt1_al_mean'] = pmt1_al / iterations
    results['pmt1_ah_mean'] = pmt1_ah / iterations
    results['pmt2_al_mean'] = pmt2_al / iterations
    results['pmt2_ah_mean'] = pmt2_ah / iterations

    return results


async def load_pmt_analog_measurement(op_id, window_ms=1.0, window_count=100):
    if (window_ms < 0.021):
        raise ValueError(f"window_ms must be greater or equal to 0.021 ms")
    if (window_count < 1) or (window_count > 65536):
        raise ValueError(f"window_count must be in the range [1, 65536]")

    fixed_range_ms = 0.02

    window_us = round((window_ms - fixed_range_ms) * 1000)

    window_coarse, window_fine = divmod(window_us, 65536)

    full_reset_delay = 40000    # 400 us
    pre_cnt_window = 100        #   1 us
    conversion_delay = 1200     #  12 us
    switch_delay = 25           # 250 ns
    reset_switch_delay = 2000   #  20 us
    input_gate_delay = 100      #   1 us
    fixed_range = round(fixed_range_ms * 1e5)

    seq_gen = meas_seq_generator()

    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 0)  # pmt1_cnt_lsb
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 1)  # pmt1_cnt_msb
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 2)  # pmt1_dt_lsb
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 3)  # pmt1_dt_msb
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 4)  # pmt1_al
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 5)  # pmt1_ah
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 6)  # pmt2_cnt_lsb
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 7)  # pmt2_cnt_msb
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 8)  # pmt2_dt_lsb
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 9)  # pmt2_dt_msb
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=10)  # pmt2_al
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=11)  # pmt2_ah
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=12)  # pmt3_cnt_lsb
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=13)  # pmt3_cnt_msb
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=14)  # pmt3_dt_lsb
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=15)  # pmt3_dt_msb

    seq_gen.ResetSignals(OutputSignal.InputGatePMT1 | OutputSignal.InputGatePMT2)
    seq_gen.SetSignals(OutputSignal.HVGatePMT1 | OutputSignal.HVGatePMT2)

    seq_gen.Loop(window_count)
    seq_gen.SetAnalogControl(pmt1=AnalogControlMode.full_offset_reset, pmt2=AnalogControlMode.full_offset_reset)
    seq_gen.ResetSignals(OutputSignal.InputGatePMT1 | OutputSignal.InputGatePMT2)

    seq_gen.TimerWaitAndRestart(full_reset_delay)
    seq_gen.SetIntegratorMode(pmt1=IntegratorMode.full_reset, pmt2=IntegratorMode.full_reset)

    seq_gen.TimerWaitAndRestart(conversion_delay)
    seq_gen.SetTriggerOutput(TriggerSignal.SamplePMT1 | TriggerSignal.SamplePMT2)

    seq_gen.TimerWaitAndRestart(switch_delay)
    seq_gen.SetIntegratorMode(pmt1=IntegratorMode.low_range_reset, pmt2=IntegratorMode.low_range_reset)
    seq_gen.SetAnalogControl(pmt1=AnalogControlMode.read_offset, pmt2=AnalogControlMode.read_offset)

    seq_gen.TimerWaitAndRestart(reset_switch_delay)
    seq_gen.SetTriggerOutput(TriggerSignal.SamplePMT1 | TriggerSignal.SamplePMT2)
    seq_gen.SetIntegratorMode(pmt1=IntegratorMode.integrate_autorange, pmt2=IntegratorMode.integrate_autorange)

    seq_gen.TimerWaitAndRestart(input_gate_delay)
    seq_gen.SetSignals(OutputSignal.InputGatePMT1 | OutputSignal.InputGatePMT2)

    seq_gen.TimerWaitAndRestart(pre_cnt_window)
    if window_coarse > 0:
        seq_gen.Loop(window_coarse)
        seq_gen.Loop(65536)
        seq_gen.TimerWaitAndRestart(pre_cnt_window)
        seq_gen.LoopEnd()
        seq_gen.LoopEnd()
    if window_fine > 0:
        seq_gen.Loop(window_fine)
        seq_gen.TimerWaitAndRestart(pre_cnt_window)
        seq_gen.LoopEnd()

    seq_gen.ResetSignals(OutputSignal.InputGatePMT1 | OutputSignal.InputGatePMT2)
    seq_gen.SetAnalogControl(pmt1=AnalogControlMode.read_offset, pmt2=AnalogControlMode.read_offset)

    seq_gen.TimerWaitAndRestart(fixed_range)
    seq_gen.SetIntegratorMode(pmt1=IntegratorMode.integrate_with_fixed_range, pmt2=IntegratorMode.integrate_with_fixed_range)

    seq_gen.TimerWaitAndRestart(conversion_delay)
    seq_gen.SetTriggerOutput(TriggerSignal.SamplePMT1 | TriggerSignal.SamplePMT2)
    seq_gen.GetAnalogResult(MeasurementChannel.PMT1, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=4)
    seq_gen.GetAnalogResult(MeasurementChannel.PMT1, isRelativeAddr=False, ignoreRange=False, isHiRange=True, addResult=True, dword=False, addrPos=0, resultPos=5)
    seq_gen.GetAnalogResult(MeasurementChannel.PMT2, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=10)
    seq_gen.GetAnalogResult(MeasurementChannel.PMT2, isRelativeAddr=False, ignoreRange=False, isHiRange=True, addResult=True, dword=False, addrPos=0, resultPos=11)

    seq_gen.LoopEnd()

    seq_gen.ResetSignals(OutputSignal.HVGatePMT1 | OutputSignal.HVGatePMT2)
    seq_gen.SetAnalogControl(pmt1=AnalogControlMode.full_offset_reset, pmt2=AnalogControlMode.full_offset_reset)
    seq_gen.SetIntegratorMode(pmt1=IntegratorMode.full_reset, pmt2=IntegratorMode.full_reset)
    seq_gen.Stop(0)

    meas_unit.resultAddresses[op_id] = range(0, 12)
    await meas_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)


async def pmt_multi_range_measurement(window_ms=1.0, iterations=100):
    
    op_id = 'pmt_multi_range_measurement'
    meas_unit.ClearOperations()
    await load_pmt_multi_range_measurement(op_id, window_ms, iterations)
        
    await meas_unit.ExecuteMeasurement(op_id)
    results = await meas_unit.ReadMeasurementValues(op_id)

    pmt1_cnt = results[0] + (results[1] << 32)
    pmt1_dt  = results[2] + (results[3] << 32)
    pmt1_al  = results[4]
    pmt1_ah  = results[5]
    pmt2_cnt = results[6] + (results[7] << 32)
    pmt2_dt  = results[8] + (results[9] << 32)
    pmt2_al  = results[10]
    pmt2_ah  = results[11]

    results = {}

    results['pmt1_cnt_mean'] = pmt1_cnt / iterations
    results['pmt1_dt_mean'] = pmt1_dt / iterations
    results['pmt1_al_mean'] = pmt1_al / iterations
    results['pmt1_ah_mean'] = pmt1_ah / iterations
    results['pmt2_cnt_mean'] = pmt2_cnt / iterations
    results['pmt2_dt_mean'] = pmt2_dt / iterations
    results['pmt2_al_mean'] = pmt2_al / iterations
    results['pmt2_ah_mean'] = pmt2_ah / iterations

    return results


async def load_pmt_multi_range_measurement(op_id, window_ms=1.0, window_count=100):
    if (window_ms < 0.001):
        raise ValueError(f"window_ms must be greater or equal to 0.001 ms")
    if (window_count < 1) or (window_count > 65536):
        raise ValueError(f"window_count must be in the range [1, 65536]")

    window_us = round(window_ms * 1000)

    window_coarse, window_fine = divmod(window_us, 65536)

    full_reset_delay = 40000    # 400 us
    pre_cnt_window = 100        #   1 us
    conversion_delay = 1200     #  12 us
    switch_delay = 25           # 250 ns
    fixed_range = 2000          #  20 us
    reset_switch_delay = 2000   #  20 us
    input_gate_delay = 100      #   1 us

    seq_gen = meas_seq_generator()

    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 0)  # pmt1_cnt_lsb
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 1)  # pmt1_cnt_msb
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 2)  # pmt1_dt_lsb
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 3)  # pmt1_dt_msb
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 4)  # pmt1_al
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 5)  # pmt1_ah
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 6)  # pmt2_cnt_lsb
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 7)  # pmt2_cnt_msb
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 8)  # pmt2_dt_lsb
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 9)  # pmt2_dt_msb
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=10)  # pmt2_al
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=11)  # pmt2_ah

    seq_gen.ResetSignals(OutputSignal.InputGatePMT1 | OutputSignal.InputGatePMT2)
    seq_gen.SetSignals(OutputSignal.HVGatePMT1 | OutputSignal.HVGatePMT2)

    seq_gen.Loop(window_count)
    seq_gen.SetAnalogControl(pmt1=AnalogControlMode.full_offset_reset, pmt2=AnalogControlMode.full_offset_reset)

    seq_gen.TimerWaitAndRestart(full_reset_delay)
    seq_gen.SetIntegratorMode(pmt1=IntegratorMode.full_reset, pmt2=IntegratorMode.full_reset)
    seq_gen.ResetSignals(OutputSignal.InputGatePMT1 | OutputSignal.InputGatePMT2)

    seq_gen.TimerWaitAndRestart(conversion_delay)
    seq_gen.SetTriggerOutput(TriggerSignal.SamplePMT1 | TriggerSignal.SamplePMT2)

    seq_gen.TimerWaitAndRestart(switch_delay)
    seq_gen.SetIntegratorMode(pmt1=IntegratorMode.low_range_reset, pmt2=IntegratorMode.low_range_reset)
    seq_gen.SetAnalogControl(pmt1=AnalogControlMode.read_offset, pmt2=AnalogControlMode.read_offset)

    seq_gen.TimerWaitAndRestart(reset_switch_delay)
    seq_gen.SetTriggerOutput(TriggerSignal.SamplePMT1 | TriggerSignal.SamplePMT2)
    seq_gen.SetIntegratorMode(pmt1=IntegratorMode.integrate_in_low_range, pmt2=IntegratorMode.integrate_in_low_range)

    seq_gen.TimerWaitAndRestart(input_gate_delay)
    seq_gen.SetSignals(OutputSignal.InputGatePMT1 | OutputSignal.InputGatePMT2)

    seq_gen.TimerWaitAndRestart(pre_cnt_window)
    seq_gen.PulseCounterControl(MeasurementChannel.PMT1, cumulative=False, resetCounter=True, resetPresetCounter=True, correctionOn=False)
    seq_gen.PulseCounterControl(MeasurementChannel.PMT2, cumulative=False, resetCounter=True, resetPresetCounter=True, correctionOn=False)
    if window_coarse > 0:
        seq_gen.Loop(window_coarse)
        seq_gen.Loop(65536)
        seq_gen.TimerWaitAndRestart(pre_cnt_window)
        seq_gen.PulseCounterControl(MeasurementChannel.PMT1, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=True)
        seq_gen.PulseCounterControl(MeasurementChannel.PMT2, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=True)
        seq_gen.LoopEnd()
        seq_gen.LoopEnd()
    if window_fine > 0:
        seq_gen.Loop(window_fine)
        seq_gen.TimerWaitAndRestart(pre_cnt_window)
        seq_gen.PulseCounterControl(MeasurementChannel.PMT1, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=True)
        seq_gen.PulseCounterControl(MeasurementChannel.PMT2, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=True)
        seq_gen.LoopEnd()

    seq_gen.ResetSignals(OutputSignal.InputGatePMT1 | OutputSignal.InputGatePMT2)
    seq_gen.SetAnalogControl(pmt1=AnalogControlMode.read_offset, pmt2=AnalogControlMode.read_offset)

    seq_gen.TimerWaitAndRestart(fixed_range)
    seq_gen.SetTriggerOutput(TriggerSignal.SamplePMT1 | TriggerSignal.SamplePMT2)
    seq_gen.SetIntegratorMode(pmt1=IntegratorMode.integrate_in_high_range, pmt2=IntegratorMode.integrate_in_high_range)
    seq_gen.GetAnalogResult(MeasurementChannel.PMT1, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=4)
    seq_gen.GetAnalogResult(MeasurementChannel.PMT2, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=10)
    
    seq_gen.TimerWaitAndRestart(conversion_delay)
    seq_gen.SetTriggerOutput(TriggerSignal.SamplePMT1 | TriggerSignal.SamplePMT2)
    seq_gen.GetAnalogResult(MeasurementChannel.PMT1, isRelativeAddr=False, ignoreRange=False, isHiRange=True, addResult=True, dword=False, addrPos=0, resultPos=5)
    seq_gen.GetAnalogResult(MeasurementChannel.PMT2, isRelativeAddr=False, ignoreRange=False, isHiRange=True, addResult=True, dword=False, addrPos=0, resultPos=11)
    
    seq_gen.GetPulseCounterResult(MeasurementChannel.PMT1, deadTime=False, relative=False, resetCounter=False, cumulative=True, dword=True, addrPos=0, resultPos=0)
    seq_gen.GetPulseCounterResult(MeasurementChannel.PMT1, deadTime=True, relative=False, resetCounter=True, cumulative=True, dword=True, addrPos=0, resultPos=2)
    seq_gen.GetPulseCounterResult(MeasurementChannel.PMT2, deadTime=False, relative=False, resetCounter=False, cumulative=True, dword=True, addrPos=0, resultPos=6)
    seq_gen.GetPulseCounterResult(MeasurementChannel.PMT2, deadTime=True, relative=False, resetCounter=True, cumulative=True, dword=True, addrPos=0, resultPos=8)

    seq_gen.LoopEnd()

    seq_gen.ResetSignals(OutputSignal.HVGatePMT1 | OutputSignal.HVGatePMT2)
    seq_gen.SetAnalogControl(pmt1=AnalogControlMode.full_offset_reset, pmt2=AnalogControlMode.full_offset_reset)
    seq_gen.SetIntegratorMode(pmt1=IntegratorMode.full_reset, pmt2=IntegratorMode.full_reset)
    seq_gen.Stop(0)

    meas_unit.resultAddresses[op_id] = range(0, 12)
    await meas_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)


async def pdd_analog_measurement(window_ms=1.0, iterations=100):
    
    op_id = 'pdd_analog_measurement'
    meas_unit.ClearOperations()
    await load_pdd_analog_measurement(op_id, window_ms, iterations)

    pmt1_pdd_scaling = config.get('pmt1_pdd_scaling', 101.0)
    pmt2_pdd_scaling = config.get('pmt2_pdd_scaling', 101.0)
    uslum_pdd_scaling = config.get('uslum_pdd_scaling', 101.0)
    htsal_pdd_scaling = config.get('htsal_pdd_scaling', 101.0)
        
    await meas_unit.ExecuteMeasurement(op_id)
    results = await meas_unit.ReadMeasurementValues(op_id)

    pmt1_pdd_al = results[0]
    pmt1_pdd_ah = results[1]
    pmt2_pdd_al = results[2]
    pmt2_pdd_ah = results[3]
    uslum_pdd_al = results[4]
    uslum_pdd_ah = results[5]
    htsal_pdd_al = results[4]
    htsal_pdd_ah = results[5]

    results = {}

    results['pmt1_pdd_al_mean'] = pmt1_pdd_al / iterations
    results['pmt1_pdd_ah_mean'] = pmt1_pdd_ah / iterations
    results['pmt1_pdd_mean'] = (pmt1_pdd_al + pmt1_pdd_ah * pmt1_pdd_scaling) / iterations
    results['pmt2_pdd_al_mean'] = pmt2_pdd_al / iterations
    results['pmt2_pdd_ah_mean'] = pmt2_pdd_ah / iterations
    results['pmt2_pdd_mean'] = (pmt2_pdd_al + pmt2_pdd_ah * pmt2_pdd_scaling) / iterations
    results['uslum_pdd_al_mean'] = uslum_pdd_al / iterations
    results['uslum_pdd_ah_mean'] = uslum_pdd_ah / iterations
    results['uslum_pdd_mean'] = (uslum_pdd_al + uslum_pdd_ah * uslum_pdd_scaling) / iterations
    results['htsal_pdd_al_mean'] = htsal_pdd_al / iterations
    results['htsal_pdd_ah_mean'] = htsal_pdd_ah / iterations
    results['htsal_pdd_mean'] = (htsal_pdd_al + htsal_pdd_ah * htsal_pdd_scaling) / iterations

    return results


async def load_pdd_analog_measurement(op_id, window_ms=1.0, window_count=100):
    if (window_ms < 0.021):
        raise ValueError(f"window_ms must be greater or equal to 0.021 ms")
    if (window_count < 1) or (window_count > 65536):
        raise ValueError(f"window_count must be in the range [1, 65536]")

    fixed_range_ms = 0.02

    window_us = round((window_ms - fixed_range_ms) * 1000)

    window_coarse, window_fine = divmod(window_us, 65536)

    full_reset_delay = 40000    # 400 us
    pre_cnt_window = 100        #   1 us
    conversion_delay = 1200     #  12 us
    switch_delay = 25           # 250 ns
    fixed_range = round(fixed_range_ms * 1e5)

    seq_gen = meas_seq_generator()

    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 0)  # ref_al
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 1)  # ref_ah
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 2)  # abs_al
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 3)  # abs_ah
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 4)  # aux_al
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 5)  # aux_ah

    seq_gen.Loop(window_count)
    seq_gen.SetAnalogControl(ref=AnalogControlMode.full_offset_reset, abs=AnalogControlMode.full_offset_reset, aux=AnalogControlMode.full_offset_reset)

    seq_gen.TimerWaitAndRestart(full_reset_delay)
    seq_gen.SetIntegratorMode(ref=IntegratorMode.full_reset, abs=IntegratorMode.full_reset, aux=IntegratorMode.full_reset)

    seq_gen.TimerWaitAndRestart(conversion_delay)
    seq_gen.SetTriggerOutput(TriggerSignal.SampleRef | TriggerSignal.SampleAbs | TriggerSignal.SampleAux)

    seq_gen.TimerWaitAndRestart(switch_delay)
    seq_gen.SetIntegratorMode(ref=IntegratorMode.low_range_reset, abs=IntegratorMode.low_range_reset, aux=IntegratorMode.low_range_reset)
    seq_gen.SetAnalogControl(ref=AnalogControlMode.read_offset, abs=AnalogControlMode.read_offset, aux=AnalogControlMode.read_offset)

    seq_gen.TimerWaitAndRestart(pre_cnt_window)
    seq_gen.SetTriggerOutput(TriggerSignal.SampleRef | TriggerSignal.SampleAbs | TriggerSignal.SampleAux)
    seq_gen.SetIntegratorMode(ref=IntegratorMode.integrate_autorange, abs=IntegratorMode.integrate_autorange, aux=IntegratorMode.integrate_autorange)

    if window_coarse > 0:
        seq_gen.Loop(window_coarse)
        seq_gen.Loop(65536)
        seq_gen.TimerWaitAndRestart(pre_cnt_window)
        seq_gen.LoopEnd()
        seq_gen.LoopEnd()
    if window_fine > 0:
        seq_gen.Loop(window_fine)
        seq_gen.TimerWaitAndRestart(pre_cnt_window)
        seq_gen.LoopEnd()

    seq_gen.SetAnalogControl(ref=AnalogControlMode.read_offset, abs=AnalogControlMode.read_offset, aux=AnalogControlMode.read_offset)

    seq_gen.TimerWaitAndRestart(fixed_range)
    seq_gen.SetIntegratorMode(ref=IntegratorMode.integrate_with_fixed_range, abs=IntegratorMode.integrate_with_fixed_range, aux=IntegratorMode.integrate_with_fixed_range)

    seq_gen.TimerWaitAndRestart(conversion_delay)
    seq_gen.SetTriggerOutput(TriggerSignal.SampleRef | TriggerSignal.SampleAbs | TriggerSignal.SampleAux)
    seq_gen.GetAnalogResult(MeasurementChannel.REF, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=0)
    seq_gen.GetAnalogResult(MeasurementChannel.REF, isRelativeAddr=False, ignoreRange=False, isHiRange=True, addResult=True, dword=False, addrPos=0, resultPos=1)
    seq_gen.GetAnalogResult(MeasurementChannel.ABS, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=2)
    seq_gen.GetAnalogResult(MeasurementChannel.ABS, isRelativeAddr=False, ignoreRange=False, isHiRange=True, addResult=True, dword=False, addrPos=0, resultPos=3)
    seq_gen.GetAnalogResult(MeasurementChannel.AUX, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=4)
    seq_gen.GetAnalogResult(MeasurementChannel.AUX, isRelativeAddr=False, ignoreRange=False, isHiRange=True, addResult=True, dword=False, addrPos=0, resultPos=5)

    seq_gen.LoopEnd()

    seq_gen.SetAnalogControl(ref=AnalogControlMode.full_offset_reset, abs=AnalogControlMode.full_offset_reset, aux=AnalogControlMode.full_offset_reset)
    seq_gen.SetIntegratorMode(ref=IntegratorMode.full_reset, abs=IntegratorMode.full_reset, aux=IntegratorMode.full_reset)
    seq_gen.Stop(0)

    meas_unit.resultAddresses[op_id] = range(0, 6)
    await meas_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)


async def pdd_multi_range_measurement(window_ms=1.0, iterations=100):
    
    op_id = 'pdd_multi_range_measurement'
    meas_unit.ClearOperations()
    await load_pdd_multi_range_measurement(op_id, window_ms, iterations)
        
    await meas_unit.ExecuteMeasurement(op_id)
    results = await meas_unit.ReadMeasurementValues(op_id)

    pmt1_pdd_al = results[0]
    pmt1_pdd_ah = results[1]
    pmt2_pdd_al = results[2]
    pmt2_pdd_ah = results[3]
    uslum_pdd_al = results[4]
    uslum_pdd_ah = results[5]
    htsal_pdd_al = results[4]
    htsal_pdd_ah = results[5]

    results = {}

    results['pmt1_pdd_al_mean'] = pmt1_pdd_al / iterations
    results['pmt1_pdd_ah_mean'] = pmt1_pdd_ah / iterations * window_ms / (window_ms + 0.02)
    results['pmt2_pdd_al_mean'] = pmt2_pdd_al / iterations
    results['pmt2_pdd_ah_mean'] = pmt2_pdd_ah / iterations * window_ms / (window_ms + 0.02)
    results['uslum_pdd_al_mean'] = uslum_pdd_al / iterations
    results['uslum_pdd_ah_mean'] = uslum_pdd_ah / iterations * window_ms / (window_ms + 0.02)
    results['htsal_pdd_al_mean'] = htsal_pdd_al / iterations
    results['htsal_pdd_ah_mean'] = htsal_pdd_ah / iterations * window_ms / (window_ms + 0.02)

    return results


async def load_pdd_multi_range_measurement(op_id, window_ms=1.0, window_count=100):
    if (window_ms < 0.001):
        raise ValueError(f"window_ms must be greater or equal to 0.001 ms")
    if (window_count < 1) or (window_count > 65536):
        raise ValueError(f"window_count must be in the range [1, 65536]")

    window_us = round(window_ms * 1000)

    window_coarse, window_fine = divmod(window_us, 65536)

    full_reset_delay = 40000    # 400 us
    pre_cnt_window = 100        #   1 us
    conversion_delay = 1200     #  12 us
    switch_delay = 25           # 250 ns
    fixed_range = 2000          #  20 us

    seq_gen = meas_seq_generator()

    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 0)  # ref_al
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 1)  # ref_ah
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 2)  # abs_al
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 3)  # abs_ah
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 4)  # aux_al
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 5)  # aux_ah

    seq_gen.Loop(window_count)
    seq_gen.SetAnalogControl(ref=AnalogControlMode.full_offset_reset, abs=AnalogControlMode.full_offset_reset, aux=AnalogControlMode.full_offset_reset)

    seq_gen.TimerWaitAndRestart(full_reset_delay)
    seq_gen.SetIntegratorMode(ref=IntegratorMode.full_reset, abs=IntegratorMode.full_reset, aux=IntegratorMode.full_reset)

    seq_gen.TimerWaitAndRestart(conversion_delay)
    seq_gen.SetTriggerOutput(TriggerSignal.SampleRef | TriggerSignal.SampleAbs | TriggerSignal.SampleAux)

    seq_gen.TimerWaitAndRestart(switch_delay)
    seq_gen.SetIntegratorMode(ref=IntegratorMode.integrate_in_low_range, abs=IntegratorMode.integrate_in_low_range, aux=IntegratorMode.integrate_in_low_range)
    seq_gen.SetAnalogControl(ref=AnalogControlMode.read_offset, abs=AnalogControlMode.read_offset, aux=AnalogControlMode.read_offset)

    seq_gen.TimerWaitAndRestart(pre_cnt_window)
    seq_gen.SetTriggerOutput(TriggerSignal.SampleRef | TriggerSignal.SampleAbs | TriggerSignal.SampleAux)

    if window_coarse > 0:
        seq_gen.Loop(window_coarse)
        seq_gen.Loop(65536)
        seq_gen.TimerWaitAndRestart(pre_cnt_window)
        seq_gen.LoopEnd()
        seq_gen.LoopEnd()
    if window_fine > 0:
        seq_gen.Loop(window_fine)
        seq_gen.TimerWaitAndRestart(pre_cnt_window)
        seq_gen.LoopEnd()

    seq_gen.SetAnalogControl(ref=AnalogControlMode.read_offset, abs=AnalogControlMode.read_offset, aux=AnalogControlMode.read_offset)

    seq_gen.TimerWaitAndRestart(fixed_range)
    seq_gen.SetTriggerOutput(TriggerSignal.SampleRef | TriggerSignal.SampleAbs | TriggerSignal.SampleAux)
    seq_gen.SetIntegratorMode(ref=IntegratorMode.integrate_in_high_range, abs=IntegratorMode.integrate_in_high_range, aux=IntegratorMode.integrate_in_high_range)
    seq_gen.GetAnalogResult(MeasurementChannel.REF, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=0)
    seq_gen.GetAnalogResult(MeasurementChannel.ABS, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=2)
    seq_gen.GetAnalogResult(MeasurementChannel.AUX, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=4)

    seq_gen.TimerWaitAndRestart(conversion_delay)
    seq_gen.SetTriggerOutput(TriggerSignal.SampleRef | TriggerSignal.SampleAbs | TriggerSignal.SampleAux)
    seq_gen.GetAnalogResult(MeasurementChannel.REF, isRelativeAddr=False, ignoreRange=False, isHiRange=True, addResult=True, dword=False, addrPos=0, resultPos=1)
    seq_gen.GetAnalogResult(MeasurementChannel.ABS, isRelativeAddr=False, ignoreRange=False, isHiRange=True, addResult=True, dword=False, addrPos=0, resultPos=3)
    seq_gen.GetAnalogResult(MeasurementChannel.AUX, isRelativeAddr=False, ignoreRange=False, isHiRange=True, addResult=True, dword=False, addrPos=0, resultPos=5)

    seq_gen.LoopEnd()

    seq_gen.SetAnalogControl(ref=AnalogControlMode.full_offset_reset, abs=AnalogControlMode.full_offset_reset, aux=AnalogControlMode.full_offset_reset)
    seq_gen.SetIntegratorMode(ref=IntegratorMode.full_reset, abs=IntegratorMode.full_reset, aux=IntegratorMode.full_reset)
    seq_gen.Stop(0)

    meas_unit.resultAddresses[op_id] = range(0, 6)
    await meas_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)


async def pdd_multi_range_measurement_seq(window_ms=1.0, iterations=100):
    
    op_id = 'pdd_multi_range_measurement_seq'
    meas_unit.ClearOperations()
    await load_pdd_multi_range_measurement_seq(op_id, window_ms, iterations)
        
    await meas_unit.ExecuteMeasurement(op_id)
    results = await meas_unit.ReadMeasurementValues(op_id)

    pmt1_pdd_al = results[0]
    pmt1_pdd_ah = results[1]
    pmt2_pdd_al = results[2]
    pmt2_pdd_ah = results[3]
    uslum_pdd_al = results[4]
    uslum_pdd_ah = results[5]
    htsal_pdd_al = results[4]
    htsal_pdd_ah = results[5]

    results = {}

    results['pmt1_pdd_al_mean'] = pmt1_pdd_al / iterations
    results['pmt1_pdd_ah_mean'] = pmt1_pdd_ah / iterations / 100
    results['pmt2_pdd_al_mean'] = pmt2_pdd_al / iterations
    results['pmt2_pdd_ah_mean'] = pmt2_pdd_ah / iterations / 100
    results['uslum_pdd_al_mean'] = uslum_pdd_al / iterations
    results['uslum_pdd_ah_mean'] = uslum_pdd_ah / iterations / 100
    results['htsal_pdd_al_mean'] = htsal_pdd_al / iterations
    results['htsal_pdd_ah_mean'] = htsal_pdd_ah / iterations / 100

    return results


async def load_pdd_multi_range_measurement_seq(op_id, window_ms=1.0, window_count=100):
    if (window_ms < 0.001):
        raise ValueError(f"window_ms must be greater or equal to 0.001 ms")
    if (window_count < 1) or (window_count > 65536):
        raise ValueError(f"window_count must be in the range [1, 65536]")

    window_us = round(window_ms * 1000)

    window_coarse, window_fine = divmod(window_us, 65536)

    full_reset_delay = 40000    # 400 us
    pre_cnt_window = 100        #   1 us
    conversion_delay = 1200     #  12 us
    switch_delay = 25           # 250 ns

    seq_gen = meas_seq_generator()

    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 0)  # ref_al
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 1)  # ref_ah
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 2)  # abs_al
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 3)  # abs_ah
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 4)  # aux_al
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 5)  # aux_ah

    seq_gen.Loop(window_count)
    seq_gen.SetAnalogControl(ref=AnalogControlMode.full_offset_reset, abs=AnalogControlMode.full_offset_reset, aux=AnalogControlMode.full_offset_reset)

    seq_gen.TimerWaitAndRestart(full_reset_delay)
    seq_gen.SetIntegratorMode(ref=IntegratorMode.full_reset, abs=IntegratorMode.full_reset, aux=IntegratorMode.full_reset)

    seq_gen.TimerWaitAndRestart(switch_delay)
    seq_gen.SetIntegratorMode(ref=IntegratorMode.integrate_in_low_range, abs=IntegratorMode.integrate_in_low_range, aux=IntegratorMode.integrate_in_low_range)

    seq_gen.TimerWaitAndRestart(pre_cnt_window)
    seq_gen.SetTriggerOutput(TriggerSignal.SampleRef | TriggerSignal.SampleAbs | TriggerSignal.SampleAux)

    if window_coarse > 0:
        seq_gen.Loop(window_coarse)
        seq_gen.Loop(65536)
        seq_gen.TimerWaitAndRestart(pre_cnt_window)
        seq_gen.LoopEnd()
        seq_gen.LoopEnd()
    if window_fine > 0:
        seq_gen.Loop(window_fine)
        seq_gen.TimerWaitAndRestart(pre_cnt_window)
        seq_gen.LoopEnd()

    seq_gen.SetAnalogControl(ref=AnalogControlMode.read_offset, abs=AnalogControlMode.read_offset, aux=AnalogControlMode.read_offset)

    seq_gen.SetTriggerOutput(TriggerSignal.SampleRef | TriggerSignal.SampleAbs | TriggerSignal.SampleAux)

    seq_gen.TimerWaitAndRestart(conversion_delay)
    seq_gen.GetAnalogResult(MeasurementChannel.REF, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=0)
    seq_gen.GetAnalogResult(MeasurementChannel.ABS, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=2)
    seq_gen.GetAnalogResult(MeasurementChannel.AUX, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=4)
    
    window_us = round(window_ms * 1000) * 100
    window_coarse, window_fine = divmod(window_us, 65536)

    seq_gen.SetAnalogControl(ref=AnalogControlMode.full_offset_reset, abs=AnalogControlMode.full_offset_reset, aux=AnalogControlMode.full_offset_reset)

    seq_gen.TimerWaitAndRestart(full_reset_delay)
    seq_gen.SetIntegratorMode(ref=IntegratorMode.full_reset, abs=IntegratorMode.full_reset, aux=IntegratorMode.full_reset)

    seq_gen.TimerWaitAndRestart(switch_delay)
    seq_gen.SetTriggerOutput(TriggerSignal.SampleRef | TriggerSignal.SampleAbs | TriggerSignal.SampleAux)
    seq_gen.SetIntegratorMode(ref=IntegratorMode.integrate_in_high_range, abs=IntegratorMode.integrate_in_high_range, aux=IntegratorMode.integrate_in_high_range)

    seq_gen.TimerWaitAndRestart(pre_cnt_window)

    if window_coarse > 0:
        seq_gen.Loop(window_coarse)
        seq_gen.Loop(65536)
        seq_gen.TimerWaitAndRestart(pre_cnt_window)
        seq_gen.LoopEnd()
        seq_gen.LoopEnd()
    if window_fine > 0:
        seq_gen.Loop(window_fine)
        seq_gen.TimerWaitAndRestart(pre_cnt_window)
        seq_gen.LoopEnd()

    seq_gen.SetAnalogControl(ref=AnalogControlMode.read_offset, abs=AnalogControlMode.read_offset, aux=AnalogControlMode.read_offset)

    seq_gen.SetTriggerOutput(TriggerSignal.SampleRef | TriggerSignal.SampleAbs | TriggerSignal.SampleAux)

    seq_gen.TimerWaitAndRestart(conversion_delay)
    seq_gen.GetAnalogResult(MeasurementChannel.REF, isRelativeAddr=False, ignoreRange=False, isHiRange=True, addResult=True, dword=False, addrPos=0, resultPos=1)
    seq_gen.GetAnalogResult(MeasurementChannel.ABS, isRelativeAddr=False, ignoreRange=False, isHiRange=True, addResult=True, dword=False, addrPos=0, resultPos=3)
    seq_gen.GetAnalogResult(MeasurementChannel.AUX, isRelativeAddr=False, ignoreRange=False, isHiRange=True, addResult=True, dword=False, addrPos=0, resultPos=5)

    seq_gen.LoopEnd()

    seq_gen.SetAnalogControl(ref=AnalogControlMode.full_offset_reset, abs=AnalogControlMode.full_offset_reset, aux=AnalogControlMode.full_offset_reset)
    seq_gen.SetIntegratorMode(ref=IntegratorMode.full_reset, abs=IntegratorMode.full_reset, aux=IntegratorMode.full_reset)
    seq_gen.Stop(0)

    meas_unit.resultAddresses[op_id] = range(0, 6)
    await meas_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)


async def set_led_current(current, source='smu', channel='led1', led_type='red'):
    if source.startswith('smu'):
        return await smu_set_led_current(current, source, channel, led_type)
    if source.startswith('fmb'):
        return await fmb_set_led_current(current, source, channel, led_type)
    
    raise Exception(f"Invalid source: {source}")


led_compliance_voltage = {'red':2.0, 'green':2.9, 'blue':3.1}

async def smu_set_led_current(current=1, source='smu', channel='led1', led_type='green'):
    if not source.startswith('smu'):
        raise ValueError(f"Invalid source: {source}")
    
    from pymeasure.instruments.keithley import Keithley2450

    led_color = led_type.split('_')[0]

    # VISA address: USB[board]::manufacturer ID::model code::serial number[::USB interface number][::INSTR]
    smu = Keithley2450('USB0::1510::9296::?*::INSTR')
    smu.apply_current()
    smu.measure_voltage()
    smu.compliance_voltage = led_compliance_voltage[led_color]
    current = round(current)
    if (current > 0) and (current <= 30000):
        smu.source_current = current * 1e-6
        smu.enable_source()
    else:
        smu.shutdown()

    await asyncio.sleep(0.2)
    return current


fmb_led_channel = {
    'led1' : {'bc':FMBAnalogOutput.BCG, 'gs':FMBAnalogOutput.GSG1},
    'led2' : {'bc':FMBAnalogOutput.BCR, 'gs':FMBAnalogOutput.GSR1},
    'led3' : {'bc':FMBAnalogOutput.BCG, 'gs':FMBAnalogOutput.GSG2},
    'led4' : {'bc':FMBAnalogOutput.BCR, 'gs':FMBAnalogOutput.GSR2},
}

fmb_full_scale_current = {
    'fmb'    :   127,   # Brightness Value [0, 127]
    'fmb#7'  :  2129,   # Measured with Keithley Multimeter (FMB#7,  24k3, Rev 2)
    'fmb#12' :  2084,   # Measured with Keithley Multimeter (FMB#12, 24k3, Rev 2)
    'fmb#15' : 11084,   # Measured with Keithley Multimeter (FMB#15,  4k7, Rev 2)
}

async def fmb_set_led_current(current=17, source='fmb', channel='led1', led_type='green'):
    bc = fmb_led_channel[channel]['bc']
    gs = fmb_led_channel[channel]['gs']
    full_scale_current = fmb_full_scale_current[source]
    brightness = current / full_scale_current
    current = round(min(int(brightness * 128), 127) / 127 * full_scale_current)
    if (current > 0):
        await fmb_endpoint.SetAnalogOutput(bc, brightness)
        await fmb_endpoint.SetAnalogOutput(gs, 1.0)
    else:
        await fmb_endpoint.SetAnalogOutput(bc, 0.0)
        await fmb_endpoint.SetAnalogOutput(gs, 0.0)

    await asyncio.sleep(0.2)
    return current


async def pmt_adc_led_power_scan(led='fmb_led1_green', current_start=1, current_stop=127, current_steps=127, logspace=0, pmt='pmt1', hv=0.35, ahs=101.0):
    on_delay  = 0.5   # Switch the LED on and wait before measuering the power
    off_delay = 0.5   # Switch the LED off after the measurement

    window_ms = 50
    iterations = 20

    GlobalVar.set_stop_gc(False)

    led_source, led_channel, led_type = led.split('_')

    if not logspace:
        current_range = np.linspace(current_start, current_stop, current_steps)
    else:
        current_range = np.logspace(np.log10(current_start), np.log10(current_stop), current_steps)

    await base_tester_enable(True)
    try:
        await pmt_set_hv(pmt, hv)
        await asyncio.sleep(pmt_set_hv_delay)
        await pmt_set_hv_enable(pmt, 1)
        await asyncio.sleep(pmt_set_hv_enable_delay)

        await send_to_gc(f"current ; power", log=True)

        power_scan = {}
        for current in current_range:

            if GlobalVar.get_stop_gc():
                return f"pmt_adc_led_power_scan stopped by user"

            await set_led_current(0, led_source, led_channel, led_type)
            await asyncio.sleep(off_delay)

            if GlobalVar.get_stop_gc():
                return f"pmt_adc_led_power_scan stopped by user"

            current = await set_led_current(current, led_source, led_channel, led_type)
            await asyncio.sleep(on_delay)

            results = await pmt_analog_measurement(window_ms, iterations)
            power_scan[current] = results[f"{pmt}_al_mean"] + results[f"{pmt}_ah_mean"] * ahs
            
            await send_to_gc(f"{current:7.0f} ; {power_scan[current]:11.2f}", log=True)

        await set_led_current(0, led_source, led_channel, led_type)

    finally:
        await pmt_set_hv_enable(pmt, 0)
        await base_tester_enable(False)

    timestamp = datetime.now().strftime('%Y%m%d_%H%M%S')
    calibration_file = f"{led}_{timestamp}.json"
    with open(f"{calibration_path}/{calibration_file}", 'w') as file:
        json.dump(power_scan, file, indent=4)

    return power_scan


async def pmt_cps_led_power_scan(led='fmb_led1_green', current_start=1, current_stop=127, current_steps=127, logspace=0, pmt='pmt1', dl=0.250, hv=0.370, ppr_ns=0.00):
    on_delay  = 0.5   # Switch the LED on and wait before measuering the power
    off_delay = 0.5   # Switch the LED off after the measurement

    window_ms = 50
    window_count = 20
    iterations = 1

    GlobalVar.set_stop_gc(False)

    led_source, led_channel, led_type = led.split('_')

    if not logspace:
        current_range = np.linspace(current_start, current_stop, current_steps)
    else:
        current_range = np.logspace(np.log10(current_start), np.log10(current_stop), current_steps)

    await base_tester_enable(True)
    try:
        await pmt_set_dl(pmt, dl)
        await pmt_set_hv(pmt, hv)
        await asyncio.sleep(pmt_set_hv_delay)
        await pmt_set_hv_enable(pmt, 1)
        await asyncio.sleep(pmt_set_hv_enable_delay)

        await send_to_gc(f"temperature: {await pmt_get_temperature(pmt)}", log=True)
        await send_to_gc(f" ")
        await send_to_gc(f"current ; power", log=True)

        power_scan = {}
        for current in current_range:

            if GlobalVar.get_stop_gc():
                return f"pmt_cps_led_power_scan stopped by user"

            await set_led_current(0, led_source, led_channel, led_type)
            await asyncio.sleep(off_delay)

            if GlobalVar.get_stop_gc():
                return f"pmt_cps_led_power_scan stopped by user"

            current = await set_led_current(current, led_source, led_channel, led_type)
            await asyncio.sleep(on_delay)

            results = await pmt_counting_measurement(window_ms, window_count, iterations)
            power_scan[current] = pmt_calculate_correction(results[f"{pmt}_cps_mean"], ppr_ns)
            
            await send_to_gc(f"{current:7.0f} ; {power_scan[current]:11.2f}", log=True)

        await set_led_current(0, led_source, led_channel, led_type)

    finally:
        await pmt_set_hv_enable(pmt, 0)
        await base_tester_enable(False)

    timestamp = datetime.now().strftime('%Y%m%d_%H%M%S')
    calibration_file = f"{led}_{timestamp}.json"
    with open(f"{calibration_path}/{calibration_file}", 'w') as file:
        json.dump(power_scan, file, indent=4)

    return power_scan


async def pdd_led_power_scan(led='fmb_led1_green', current_start=1, current_stop=127, current_steps=127, logspace=0, channel='pmt1'):
    on_delay  = 0.5   # Switch the LED on and wait before measuering the power
    off_delay = 0.5   # Switch the LED off after the measurement

    window_ms = 49.5
    iterations = 20

    GlobalVar.set_stop_gc(False)

    led_source, led_channel, led_type = led.split('_')

    if not logspace:
        current_range = np.linspace(current_start, current_stop, current_steps)
    else:
        current_range = np.logspace(np.log10(current_start), np.log10(current_stop), current_steps)

    await base_tester_enable(True)
    try:
        await send_to_gc(f"current ; power", log=True)

        power_scan = {}
        for current in current_range:

            if GlobalVar.get_stop_gc():
                return f"pdd_led_power_scan stopped by user"

            await set_led_current(0, led_source, led_channel, led_type)
            await asyncio.sleep(off_delay)

            if GlobalVar.get_stop_gc():
                return f"pdd_led_power_scan stopped by user"

            current = await set_led_current(current, led_source, led_channel, led_type)
            await asyncio.sleep(on_delay)

            results = await pdd_analog_measurement(window_ms, iterations)
            power_scan[current] = results[f"{channel}_pdd_mean"]
            
            await send_to_gc(f"{current:7.0f} ; {power_scan[current]:11.2f}", log=True)

        await set_led_current(0, led_source, led_channel, led_type)

    finally:
        await base_tester_enable(False)

    timestamp = datetime.now().strftime('%Y%m%d_%H%M%S')
    calibration_file = f"{led}_{timestamp}.json"
    with open(f"{calibration_path}/{calibration_file}", 'w') as file:
        json.dump(power_scan, file, indent=4)

    return power_scan


async def opm_led_power_scan(led='fmb_led1_green', current_start=1, current_stop=127, current_steps=127, logspace=0):
    on_delay  = 5   # Switch the LED on and wait before measuering the power
    off_delay = 5   # Switch the LED off after the measurement

    GlobalVar.set_stop_gc(False)

    led_source, led_channel, led_type = led.split('_')

    if not logspace:
        current_range = np.linspace(current_start, current_stop, current_steps)
    else:
        current_range = np.logspace(np.log10(current_start), np.log10(current_stop), current_steps)

    await base_tester_enable(True)
    try:
        opm_info = opm_init(led_type)
        await send_to_gc(f"{opm_info}", log=True)

        await send_to_gc(f"current ; power", log=True)

        power_scan = {}
        for current in current_range:

            if GlobalVar.get_stop_gc():
                return f"opm_led_power_scan stopped by user"

            await set_led_current(0, led_source, led_channel, led_type)
            await asyncio.sleep(off_delay)

            if GlobalVar.get_stop_gc():
                return f"opm_led_power_scan stopped by user"

            current = await set_led_current(current, led_source, led_channel, led_type)
            await asyncio.sleep(on_delay)

            power_scan[current] = np.round(opm_get_power() * 1e9, decimals=2)
            
            await send_to_gc(f"{current:7.0f} ; {power_scan[current]:11.2f}", log=True)

        await set_led_current(0, led_source, led_channel, led_type)

    finally:
        await base_tester_enable(False)

    timestamp = datetime.now().strftime('%Y%m%d_%H%M%S')
    calibration_file = f"{led}_{timestamp}_filter_{opm_info['filter']}.json"
    with open(f"{calibration_path}/{calibration_file}", 'w') as file:
        json.dump(power_scan, file, indent=4)

    return power_scan


led_wavelength = {'red':635, 'green':525, 'blue':465}

opm_serial_port = '/dev/ttyUSB0'
opm_bautrate = 115200

def opm_init(wavelength='red'):
    if wavelength in led_wavelength:
        wavelength = led_wavelength[wavelength]

    with Serial(opm_serial_port, opm_bautrate, timeout=2) as serial:
        idn = opm_send_command(serial, f"II")
        if (idn.split(' ')[0] != 'STBR'):
            raise Exception(f"Failed to read power meter identification: {idn}")
        
        opm_send_command(serial, f"FP")                 # Power Measurement
        opm_send_command(serial, f"WN -1")              # Autorange
        opm_send_command(serial, f"WL {wavelength}")    # Wavelength
        opm_send_command(serial, f"AQ 1")               # No Average

        filter = opm_send_command(serial, f"FQ").split(' ')[2]

    return {'idn':idn, 'wavelength':wavelength, 'filter':filter}


def opm_get_power() -> float:
    with Serial(opm_serial_port, opm_bautrate, timeout=2) as serial:
        response = opm_send_command(serial, f"SP")
        if response.lower() == 'over':
            raise Exception(f"Overrange error")
        power = float(response)
        return power


def opm_send_command(serial: Serial, command: str) -> str:
    serial.write(f'${command}\r\n'.encode('utf-8'))
    response = serial.readline().decode('utf-8')
    if not response.startswith('*'):
        raise Exception(f"Invalid power meter command: {command} -> {response}")
    
    return response[1:-2]


async def pmt_adjust_normalization(scan_type='default', reference_power_scan='pmt1_reference_power_scan', led_power_scan='fmb_led1_green_20240124_130402_filter_OUT', min_current=0):

    await send_to_gc(f"pmt_adjust_normalization(scan_type={scan_type}, led_power_scan={led_power_scan}, reference_power_scan={reference_power_scan}, min_current={min_current})", log=True)
    await send_to_gc(f" ")

    sample_config = f"pmt_signal_scan_{scan_type}"
    sample_points = config[sample_config]['led_current_bright']
    await send_to_gc(f"config[{sample_config}]['led_current_bright'] = {sample_points}", log=True)
    await send_to_gc(f" ")

    with open(f"{calibration_path}/{led_power_scan}.json", mode='r') as file:
        power_scan: dict = json.load(file)
        currents = np.array([int(key) for key in power_scan.keys()])
        powers = np.array([float(value) for value in power_scan.values()])
        await send_to_gc(f"led_power_scan = {power_scan}", log=True)

    with open(f"{calibration_path}/{reference_power_scan}.json", mode='r') as file:
        reference_scan: dict = json.load(file)
        await send_to_gc(f"reference_power_scan = {reference_scan}", log=True)

    await send_to_gc(f" ", log=True)

    normalization = np.zeros_like(sample_points)
    normalization[0] = max(sample_points[0], min_current)   # first sample point is the fixed minimum for the normalization

    await send_to_gc(f"current ; power   ; target  ; error", log=True)

    for i in range(1, len(sample_points)):
        power_target = reference_scan[str(sample_points[i])]
        current = currents[np.abs(powers - power_target).argmin()]
        normalization[i] = current if (current > normalization[i - 1]) else (normalization[i - 1] + 1)
        
        power = power_scan[str(normalization[i])]
        error = abs(power - power_target) / power_target
        await send_to_gc(f"{current:7.0f} ; {power:7.2f} ; {power_target:7.2f} ; {error:7.2%} ; ", log=True)

    await send_to_gc(f" ", log=True)

    return f"normalization = {[value for value in normalization]}"

